package com.instabug.commons.snapshot

import android.content.Context
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.lifecycle.CompositeLifecycleOwner
import com.instabug.commons.lifecycle.CompositeLifecycleOwner.CompositeLifecycleObserver
import com.instabug.commons.logging.getOrReportError
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.utils.generateReproConfigsMap
import com.instabug.commons.utils.updateScreenShotAnalytics
import com.instabug.library.model.State
import com.instabug.library.util.threading.defensiveLog
import com.instabug.library.util.threading.reportOOM
import java.io.File
import java.nio.charset.StandardCharsets
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

/**
 * Generic contract for captors.
 * A captor represents a periodic capturing mechanism for files, database updates, etc ...
 */
interface Captor {
    /**
     * Returns the id of the captor. Should be unique and saved for further use by the [CaptorsRegistry].
     */
    val id: Int

    /**
     * Indicates whether the captor has been shutdown or not.
     */
    val isShutdown: Boolean

    /**
     * Starts the captor with either a preconfigured period or period configured upon initialization.
     */
    fun start()

    /**
     * Shuts down the captor for good. Shutting down the captor prevents further start commands.
     */
    fun shutdown()

    /**
     * Forces the captor to execute its preconfigured job immediately.
     */
    fun force()
}

abstract class AbstractCaptor(
    protected val scheduler: ScheduledExecutorService
) : Captor {
    protected abstract val capturingPeriod: Long

    private var scheduledJob: ScheduledFuture<*>? = null

    @Volatile
    final override var isShutdown: Boolean = false

    private val isRunning: Boolean
        get() = !(scheduledJob?.isCancelled ?: true)

    protected abstract fun capture()

    protected open fun onStart() {
        //To be implemented for on start operations
    }

    protected open fun onShutdown() {
        //To be implemented for on shut operations
    }

    final override fun start() = synchronized(this) {
        if (!internalStart(0)) return
        onStart()
    }

    final override fun shutdown(): Unit = synchronized(this) {
        if (isShutdown) return
        runCatching(this::onShutdown)
        runCatching {
            internalPause()
            isShutdown = true
        }
    }

    final override fun force(): Unit = synchronized(this) {
        if (isShutdown) return
        runCatching {
            internalPause()
            scheduler.execute { capture(); internalStart(capturingPeriod) }
        }
    }

    protected fun internalStart(withDelay: Long): Boolean {
        if (isRunning || isShutdown) return false
        scheduledJob = scheduler.scheduleCaptor(this::captureSafely, withDelay)
        return true
    }

    private fun captureSafely() {
        runCatching(this::capture).onFailure { error ->
            if (error is InterruptedException) {
                throw error
            } else {
                defensiveLog(error)
                error.takeIf { it is OutOfMemoryError }
                    ?.let { it as OutOfMemoryError }
                    ?.also { reportOOM(it) }
            }
        }.also { scheduledJob = scheduler.scheduleCaptor(this::captureSafely, capturingPeriod) }
    }

    protected fun internalPause(): Boolean {
        if (!isRunning || isShutdown) return false
        scheduledJob?.cancel(true)
        scheduledJob = null
        return true
    }

    private fun ScheduledExecutorService.scheduleCaptor(
        captor: Runnable,
        delay: Long = 0L
    ): ScheduledFuture<*>? = run {
        if (Thread.currentThread().isInterrupted) return@run null
        schedule(captor, delay, TimeUnit.SECONDS)
    }
}

class CaptorConfigurations(
    private val ctxGetter: () -> Context?,
    private val savingDirectoryGetter: () -> File?,
    val scheduler: ScheduledExecutorService
) {
    val ctx: Context?
        get() = ctxGetter()
    val savingDirectory: File?
        get() = savingDirectoryGetter()
}

class StateSnapshotCaptor(
    private val configurations: CaptorConfigurations,
    private val lifecycleOwner: CompositeLifecycleOwner,
) : AbstractCaptor(configurations.scheduler), CompositeLifecycleObserver {
    override val id: Int
        get() = ID
    override val capturingPeriod: Long
        get() = 5L

    private val File.snapshotFile: File
        get() = File("$absolutePath${File.separator}$STATE_SNAPSHOT_FILE_NAME_JSON")

    private val File.oldSnapshotFile: File
        get() = File("$absolutePath${File.separator}$OLD_STATE_SNAPSHOT_FILE_NAME_JSON")

    override fun onStart() {
        lifecycleOwner.register(this)
        "Starting state snapshot captor".logVerbose()
    }

    override fun onShutdown() {
        lifecycleOwner.unregister(this)
        "Shutting down state snapshot captor".logVerbose()
    }

    override fun capture() {
        if (Thread.currentThread().isInterrupted) return
        configurations.savingDirectory?.run {
            //Rename old snapshot file to be suffixed with -old
            snapshotFile.takeIf { it.exists() }
                ?.also { it.rename(oldSnapshotFile.name) }

            snapshotFile.parentFile?.ifNotExists { mkdirs() }
            snapshotFile.ifNotExists { createNewFile() }
            //Create new snapshot
            configurations.ctx?.let { ctx ->
                State.Builder(ctx)
                    .build(true, true, true, 1.0f, false)
                    .apply(State::updateVisualUserSteps)
                    .apply(State::generateReproConfigsMap)
                    .apply(this@StateSnapshotCaptor::updateScreenShotAnalyticsData)
                    .toJson()
                    .let(snapshotFile::writeTextSafely)
            }

            //Delete old snapshot
            oldSnapshotFile.takeIf { it.exists() }?.delete()
        }
    }

    private fun updateScreenShotAnalyticsData(state: State): State =
        state.apply {
            this.updateScreenShotAnalytics()
        }


    override fun onActivityStarted() {
        "StateSnapshotCaptor: Activity started".logVerbose()
        force()
    }

    override fun onFragmentStarted() {
        "StateSnapshotCaptor: Fragment started".logVerbose()
        force()
    }

    object Factory {
        @JvmStatic
        @JvmOverloads
        operator fun invoke(
            ctxGetter: () -> Context? = CommonsLocator::appCtx,
            savingDirectoryGetter: () -> File? = CommonsLocator.crashesCacheDir::currentSessionDirectory,
            scheduler: ScheduledExecutorService = CommonsLocator.getCoreScheduler(),
            lifecycleOwner: CompositeLifecycleOwner = CommonsLocator.compositeLifecycleOwner,
        ) = StateSnapshotCaptor(
            configurations = CaptorConfigurations(
                ctxGetter,
                savingDirectoryGetter,
                scheduler
            ),
            lifecycleOwner = lifecycleOwner
        )
    }

    companion object {
        const val ID = 0x001
        private const val STATE_SNAPSHOT_FILE_NAME = "snapshot"
        const val STATE_SNAPSHOT_FILE_NAME_JSON = "snapshot.json"
        private const val OLD_STATE_SNAPSHOT_FILE_SUFFIX = "-old"
        const val OLD_STATE_SNAPSHOT_FILE_NAME_JSON = "snapshot-old.json"
        private const val CAPTOR_NAME = "CrashesStateSnapshot"

        private fun getSnapshotFile(sessionDirectory: File): File =
            sessionDirectory.run {
                File("$absolutePath${File.separator}$STATE_SNAPSHOT_FILE_NAME")
            }

        private fun getJsonSnapshotFile(sessionDirectory: File): File =
            sessionDirectory.run {
                File("$absolutePath${File.separator}${STATE_SNAPSHOT_FILE_NAME_JSON}")
            }

        private fun getOldJsonSnapshotFile(sessionDirectory: File): File =
            sessionDirectory.run {
                File("${sessionDirectory.absolutePath}${File.separator}$OLD_STATE_SNAPSHOT_FILE_NAME_JSON")
            }

        private fun getOldSnapshotFile(sessionDirectory: File): File =
            getSnapshotFile(sessionDirectory).run { File("$absolutePath$OLD_STATE_SNAPSHOT_FILE_SUFFIX") }

        fun getStateSnapshotFile(sessionDirectory: File): File =
            getOldSnapshotFile(sessionDirectory).takeIf { it.exists() }
                ?: getSnapshotFile(sessionDirectory).takeIf { it.exists() }
                ?: getOldJsonSnapshotFile(sessionDirectory).takeIf { it.exists() }
                ?: getJsonSnapshotFile(sessionDirectory)


        @JvmStatic
        fun getStateSnapshot(sessionDirectory: File): State? {
            val snapshotFile: File = getStateSnapshotFile(sessionDirectory)
            var state: State? = null
            snapshotFile.run {
                if (absolutePath.endsWith(".json")) {
                    runCatching {
                        val jsonString = readText(StandardCharsets.UTF_8)
                        jsonString.takeIf { it.isNotEmpty() }
                            ?.let {
                                state = State().apply { fromJson(jsonString) }
                            }
                    }.getOrReportError(null, "Error while reading state json file.")
                } else {
                    state = readSerializable()
                }
            }
            return state
        }
    }
}
