package com.instabug.bganr

import androidx.annotation.VisibleForTesting
import com.instabug.commons.di.CommonsLocator
import org.json.JSONArray
import org.json.JSONObject
import java.io.InputStream
import java.lang.Thread.State
import java.util.regex.Matcher

private const val REGEX_DALVIK_THREADS_ENTRY = "DALVIK THREADS \\(\\d*\\):"
private const val REGEX_DALVIK_THREADS_EXIT = "----- end \\d* -----"
private const val REGEX_THREAD_ATTRS =
    "^\"(.*)\"(?: daemon)*(?: prio=(\\d+))*(?: tid=(\\d+))*(?: ([a-zA-Z]+))*(?: .*)*"
private const val REGEX_FRAME = " {2}at (.*\\((.*)\\))"
private const val REGEX_FILE_AND_LINE = "(.*):(.*)"
private const val REGEX_GROUP_NAME = "group=\"(.*)\""

private const val STACK_TRACE_ELEMENT = "\t at %s\n"

private const val KEY_THREAD = "thread"
private const val KEY_THREAD_NAME = "threadName"
private const val KEY_THREAD_ID = "threadId"
private const val KEY_THREAD_PRIORITY = "threadPriority"
private const val KEY_THREAD_STATE = "threadState"
private const val KEY_THREAD_GROUP = "threadGroup"
private const val KEY_IS_MAIN = "isMain"
private const val KEY_IS_CRASHING = "isCrashing"
private const val KEY_STACK_TRACE = "stackTrace"
private const val KEY_DROPPED_THREADS = "droppedThreads"
private const val KEY_DROPPED_FRAMES = "droppedFrames"
private const val KEY_TERMINATED_THREADS = "terminatedThreads"
private const val KEY_NAME = "name"
private const val KEY_EXCEPTION = "exception"
private const val KEY_LOCATION = "location"
private const val KEY_ERROR = "error"

class BackgroundAnrTraceFileParser {

    @JvmOverloads
    operator fun invoke(
        traceStream: InputStream,
        message: String,
        exception: String,
        threadLimit: Int = CommonsLocator.threadingLimitsProvider.provideThreadsLimit(),
        framesLimit: Int = CommonsLocator.threadingLimitsProvider.provideFramesLimit()
    ): Pair<JSONObject, JSONArray> {
        val aggregator = ThreadsAggregator(threadLimit, framesLimit, message, exception)
        ThreadBlocksStream().invoke(traceStream)
            .filterNot(String::isBlank)
            .map(::ThreadObject)
            .forEach(aggregator::processNewThread)
        return aggregator.finish()
    }
}

/**
 * Responsible for turning an [InputStream] of a trace file to a sequence of thread blocks.
 * An instance of this class processes the trace file line by line constructing
 * thread blocks. Threads region in a standard trace file starts with "DALVIK THREADS \n"
 * and ends with "----- end {process_id} -----".
 * Each thread block starts with a standard attributes header and ends with a new line.
 */
@VisibleForTesting
class ThreadBlocksStream {

    private val threadBlockBuilder = StringBuilder()

    /**
     * Reads a trace's [InputStream] line by line and Transforms it to a [Sequence] of thread block [String]s.
     * Threads region in a standard trace file starts with "DALVIK THREADS \n"
     * and ends with "----- end {process_id} -----". Each thread block starts with a standard
     * attributes header and ends with a new line.
     * Non-standard trace files (corrupt) will result in an unexpected result.
     * The [InputStream] given will be closed once the [Sequence] is generated. So, the generated
     * sequence is constraint to be consumed once.
     * @param traceStream A trace file's [InputStream]
     * @return a [Sequence] of [String]s each represents a whole thread.
     */
    operator fun invoke(traceStream: InputStream): Sequence<String> = sequence {
        traceStream.bufferedReader().use { reader ->
            reader.lineSequence()
                // Ignores all lines before "DALVIK THREADS"
                .dropWhile { line -> !line.matches(REGEX_DALVIK_THREADS_ENTRY.toRegex()) }
                // Ignores all lines after "----- end {pid} -----"
                .takeWhile { line -> !line.matches(REGEX_DALVIK_THREADS_EXIT.toRegex()) }
                // First line after drops and takes is "DALVIK THREADS", we need to filter that out.
                .filterNot { line -> line.matches(REGEX_DALVIK_THREADS_ENTRY.toRegex()) }
                // Append each line after filters to thread block.
                .onEach(threadBlockBuilder::appendLine)
                // If the line in hand is blank, a thread block is finalized
                .forEach { line -> finalizeBlockIfPossible(line) }
        }
    }.constrainOnce()

    private suspend fun SequenceScope<String>.finalizeBlockIfPossible(line: String) {
        if (line.isNotBlank()) return
        yield(threadBlockBuilder.toString())
        threadBlockBuilder.clear()
    }
}

/**
 * A friendly representation of a thread block.
 * @constructor Creates an instance processing a given a [String] representation of a thread.
 */
@VisibleForTesting
class ThreadObject(private val threadBlock: String) {

    private val attrsMatcher: Matcher
            by lazy { REGEX_THREAD_ATTRS.toPattern().matcher(threadBlock).apply(Matcher::find) }

    /**
     * Whether the thread in hand is the main thread.
     * If the thread name is anything other than "main" returns false, else true
     */
    val isMain: Boolean
        get() = attrsMatcher.runCatching {
            group(1)?.equals("main", true) ?: false
        }.getOrElse { false }

    /**
     * Whether the thread in hand is terminated.
     * If thread state is anything other than [Thread.State.TERMINATED] returns false, else true
     */
    val isTerminated
        get() = attrsMatcher.runCatching {
            group(4)?.equals(State.TERMINATED.name, ignoreCase = true) ?: false
        }.getOrElse { false }

    private val framesAndLocationSequence: Sequence<Pair<String, String?>>
        get() = sequence {
            REGEX_FRAME.toPattern().matcher(threadBlock)
                .runCatching { while (find()) yield(STACK_TRACE_ELEMENT.format(group(1)) to group(2)) }
        }

    /**
     * Returns the thread in hand represented as a thread's array element [JSONObject] as follows:
     * {
     *  thread": {
     *      "threadName": String,,
     *      "treadId": Int,
     *      "threadPriority": Int,
     *      "threadState": String,
     *      "threadGroup": {
     *          "name": String
     *  },
     *  "isMain": Boolean,
     *  "isCrashing": Boolean,
     *  "stackTrace": String,
     *  "droppedFrames": Int
     * }
     * @param isTrueMain If there's an indicator that this thread shouldn't be treated as main in [isMain].
     * @param framesLimit The frames limit that should be applied to this thread.
     * @return [JSONObject] representation of the thread as a thread's array element.
     */
    fun getAsArrayElement(isTrueMain: Boolean, framesLimit: Int) = JSONObject().apply {
        getThreadMetaObject().apply {
            val (stackTrace, droppedFramesCount) =
                getStackTraceAndDroppedCount(framesLimit = framesLimit)
            put(KEY_IS_MAIN, (isMain && isTrueMain))
            put(KEY_IS_CRASHING, false)
            put(KEY_STACK_TRACE, stackTrace)
            put(KEY_DROPPED_FRAMES, droppedFramesCount)
        }.also { put(KEY_THREAD, it) }
    }

    /**
     * Returns the thread in hand represented as a details [JSONObject] as follows
     * {
     *  "thread": {
     *      "threadName": String,
     *      "threadId": Int,
     *      "threadPriority": Int,
     *      "threadState": String,
     *      "threadGroup": {
     *          "name": String
     *      }
     *  },
     *  "error": {
     *      "name": String,
     *      "exception": String,
     *      "location": String,
     *      "stackTrace": String
     *  },
     *  "droppedThreads": Int,
     *  "terminatedThreads": Int
     * }
     * @param message the message that should be placed in the error.name attr.
     * @param exception the exception that should be placed in error.exception attr and concatenated
     * to the start of the error.stackTrace attr.
     * @return [JSONObject] representation of the thread as a details object.
     */
    fun getAsDetailsObject(message: String, exception: String) = JSONObject().apply {
        put(KEY_THREAD, getThreadMetaObject())
        JSONObject().apply {
            put(KEY_NAME, message)
            put(KEY_EXCEPTION, exception)
            put(KEY_STACK_TRACE, getStackTraceAndDroppedCount(exception = exception).first)
            getErrorLocation()?.let { put(KEY_LOCATION, it) }
        }.also { put(KEY_ERROR, it) }
    }

    private fun getThreadMetaObject(): JSONObject = JSONObject().apply {
        REGEX_THREAD_ATTRS.toPattern().matcher(threadBlock).apply(Matcher::find)
            .runCatching {
                group(1)?.let { put(KEY_THREAD_NAME, it) }
                group(3)?.toLongOrNull()?.let { put(KEY_THREAD_ID, it) }
                group(2)?.toIntOrNull()?.let { put(KEY_THREAD_PRIORITY, it) }
                group(4)?.let { put(KEY_THREAD_STATE, it) }
            }
        REGEX_GROUP_NAME.toPattern().matcher(threadBlock).apply(Matcher::find)
            .runCatching {
                group(1)?.let { threadGroup ->
                    JSONObject().apply { put(KEY_NAME, threadGroup) }
                }.also { put(KEY_THREAD_GROUP, it) }
            }
    }

    private fun getStackTraceAndDroppedCount(
        framesLimit: Int = Int.MAX_VALUE,
        exception: String? = null
    ): Pair<String, Int> {
        var framesCount = 0
        val stacktraceBuilder = StringBuilder().apply {
            exception?.let(this::appendLine)
            framesAndLocationSequence.onEach { framesCount++ }
                .filter { framesCount <= framesLimit }
                .forEach { (frame, _) -> append(frame) }
        }
        val droppedFramesCount = (framesCount - framesLimit).takeIf { it > 0 } ?: 0
        return stacktraceBuilder.toString() to droppedFramesCount
    }

    private fun getErrorLocation(): String? {
        val fileAndLine = framesAndLocationSequence.firstOrNull()?.second ?: return null
        if (fileAndLine.matches(REGEX_FILE_AND_LINE.toRegex())) return fileAndLine
        val isNativeMethod = fileAndLine.equals("Native Method", ignoreCase = true)
        val lineNumber = if (isNativeMethod) -2 else -1
        return "$fileAndLine:$lineNumber"
    }
}

/**
 * Responsible for aggregating threads data.
 * @constructor Creates a new instance of the aggregator given: thread limits to follow,
 * frames limit per thread.
 */
private class ThreadsAggregator(
    private val threadsLimit: Int,
    private val framesLimitPerThread: Int,
    private val message: String,
    private val exception: String
) {
    private var isMainProcessed = false
    private var totalThreadsCount = 0
    private var terminatedThreadsCount = 0
    private val threadsArray: JSONArray = JSONArray()
    private var detailsObject: JSONObject = JSONObject()

    private val isThreadsLimitExceeded: Boolean
        get() {
            val comparableLimit = if (isMainProcessed) threadsLimit else (threadsLimit - 1)
            return threadsArray.length() >= comparableLimit
        }

    /**
     * Processes a new [ThreadObject] fulfilling certain rules for aggregation:
     * If the thread is terminated and not the main thread -> increment terminated count.
     * If the thread is the main thread -> add it to the threads array and construct the details object.
     * If there's a room for one more thread (following limits) -> add the thread to threads array.
     * Also, only the first main thread as defined by [ThreadObject] shall be considered a true main,
     * any other seemingly a main will be treated as any other thread.
     * @param threadObject the [ThreadObject] representation of a given thread.
     */
    fun processNewThread(threadObject: ThreadObject) {
        totalThreadsCount++
        val isMainThread = threadObject.isMain && !isMainProcessed
        val isTerminated = threadObject.isTerminated
        if (isTerminated && !isMainThread) {
            terminatedThreadsCount++
            return
        }
        if (!isMainThread && isThreadsLimitExceeded) return
        threadsArray.put(threadObject.getAsArrayElement(!isMainProcessed, framesLimitPerThread))
        if (!isMainThread) return
        detailsObject = threadObject.getAsDetailsObject(message, exception)
        isMainProcessed = true
    }

    /**
     * Aggregates the results of processed threads.
     * Calling this marks the finish of processing any new threads. So, the aggregate counts is being added
     * to the details object at this point.
     * @return a [Pair] of [JSONObject] and [JSONArray] represents the details object and the threads array respectively.
     */
    fun finish(): Pair<JSONObject, JSONArray> {
        val droppedThreadsCount = totalThreadsCount - terminatedThreadsCount - threadsArray.length()
        detailsObject.apply {
            put(KEY_DROPPED_THREADS, droppedThreadsCount)
            put(KEY_TERMINATED_THREADS, terminatedThreadsCount)
        }
        return detailsObject to threadsArray
    }
}
