package com.instabug.anr.configuration

import com.instabug.anr.di.AnrServiceLocator
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.preferences.crashesPreferences
import com.instabug.commons.utils.SharedPreferencesUtils
import com.instabug.crash.Constants.Preferences.ANR_AVAILABILITY
import com.instabug.crash.Constants.Preferences.IS_ANR_MIGRATED
import com.instabug.library.Instabug
import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.internal.sharedpreferences.corePreferences
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject

class AnrConfigurationHandlerImpl : ConfigurationsHandler {

    companion object {
        private const val CRASHES_RESPONSE_KEY = "crashes"
        private const val ANR_RESPONSE_KEY = "anr"
        private const val ANR_V2_RESPONSE_KEY = "anr_v2"
        const val CORE_ANR_KEY = "ANR_REPORTINGAVAIL"
    }

    override fun handleConfiguration(configuration: String?) {
        runOrLogError(errorMessage = "Something went wrong while parsing ANR from features response ") {
            configuration?.let {
                val featuresResponse = JSONObject(configuration)
                featuresResponse.optJSONObject(CRASHES_RESPONSE_KEY)
                    ?.let {
                        it.optBoolean(ANR_RESPONSE_KEY).also { isAvailable ->
                            updateAnrAvailability(isAvailable)
                        }
                        it.optBoolean(ANR_V2_RESPONSE_KEY).let(::updateAnrV2Availability)
                    }

            }
        }
    }

    override fun migrateCurrentConfiguration() {
        if (!isAnrMigrated())
            Instabug.getApplicationContext()?.let {
                SharedPreferencesUtils.readBoolean(
                    CORE_ANR_KEY,
                    ANR_AVAILABILITY.second,
                    corePreferences
                ).also { migratedValue ->
                    updateAnrAvailability(migratedValue)
                    setAnrMigrated()
                }
            }
    }

    override fun handle(modesMap: Map<Int, Int>) {
        modesMap[IssueType.ANR]?.let { anrsMode ->
            with(AnrServiceLocator.anrConfigurationProvider) {
                isReproStepsEnabledSDK = anrsMode > ReproMode.Disable
                isReproScreenShotsEnabledSDK = anrsMode > ReproMode.EnableWithNoScreenshots
            }
        }
    }

    private fun updateAnrAvailability(isAvailable: Boolean) {
        AnrServiceLocator.anrConfigurationProvider.isAnrAvailable = isAvailable
    }

    private fun updateAnrV2Availability(isAvailable: Boolean) {
        AnrServiceLocator.anrConfigurationProvider.isAnrV2Available = isAvailable
    }

    private fun isAnrMigrated(): Boolean {
        val (key, defaultValue) = IS_ANR_MIGRATED
        return SharedPreferencesUtils.readBoolean(key, defaultValue, crashesPreferences)
    }

    private fun setAnrMigrated() {
        SharedPreferencesUtils.writeBoolean(
            IS_ANR_MIGRATED.first,
            true,
            crashesPreferences
        )
    }
}