package com.instabug.terminations.configuration

import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.preferences.crashesPreferences
import com.instabug.commons.utils.SharedPreferencesUtils
import com.instabug.crash.Constants
import com.instabug.crash.Constants.Preferences.TERMINATIONS_AVAILABILITY
import com.instabug.crash.Constants.Preferences.TERMINATIONS_STATE_RATIO
import com.instabug.crash.Constants.Preferences.TERMINATIONS_THRESHOLD
import com.instabug.library.Instabug
import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.visualusersteps.ReproRuntimeConfigurationsHandler
import com.instabug.terminations.di.ServiceLocator.terminationsConfigurationProvider
import org.json.JSONObject

class TerminationsConfigurationHandler : ConfigurationsHandler, ReproRuntimeConfigurationsHandler {

    companion object {
        private const val CONFIG_KEY_CRASHES = "crashes"
        private const val CONFIG_KEY_TERMINATIONS = "android_user_termination"
        private const val CONFIG_KEY_ENABLED = "enabled"
        private const val CONFIG_KEY_THRESHOLD = "time_between_sessions"
        private const val CONFIG_KEY_STATE_RATIO = "logs_percentage"
    }

    override fun handleConfiguration(configuration: String?) {
        if (!configurationsParsed(configuration)) {
            terminationsConfigurationProvider.isAvailable = TERMINATIONS_AVAILABILITY.second
        }
    }

    private fun configurationsParsed(configuration: String?): Boolean {
        runOrLogError("Something went wrong while parsing App terminations from features response ") {
            configuration?.let {
                getTerminationsObject(it)?.let { termination ->
                    with(terminationsConfigurationProvider) {
                        isAvailable = termination.optAvailability()
                        threshold = termination.optThreshold()
                        stateRatio = termination.optStateRatio().toFloat()
                        return true
                    }
                }
            }
        }
        return false
    }

    override fun handle(modesMap: Map<Int, Int>) {
        modesMap[IssueType.ForceRestart]?.let { restartsMode ->
            with(terminationsConfigurationProvider) {
                isReproStepsEnabledSDK = restartsMode > ReproMode.Disable
                isReproScreenShotsEnabledSDK = restartsMode > ReproMode.EnableWithNoScreenshots
            }
        }
    }

    private fun getTerminationsObject(configString: String): JSONObject? = kotlin.runCatching {
        JSONObject(configString)
            .getJSONObject(CONFIG_KEY_CRASHES)
            .getJSONObject(CONFIG_KEY_TERMINATIONS)
    }.getOrNull()

    private fun JSONObject.optAvailability() =
        optBoolean(
            CONFIG_KEY_ENABLED,
            TERMINATIONS_AVAILABILITY.second
        )

    private fun JSONObject.optThreshold() =
        optLong(
            CONFIG_KEY_THRESHOLD,
            TERMINATIONS_THRESHOLD.second
        )

    private fun JSONObject.optStateRatio() =
        optDouble(
            CONFIG_KEY_STATE_RATIO,
            TERMINATIONS_STATE_RATIO.second.toDouble()
        )

    override fun migrateCurrentConfiguration() {
        takeUnless { isAppTerminationsMigrated() }
            ?.let {
                with(terminationsConfigurationProvider) {
                    Instabug.getApplicationContext()?.let {
                        SharedPreferencesUtils.readBoolean(
                            CONFIG_KEY_ENABLED,
                            TERMINATIONS_AVAILABILITY.second,
                            crashesPreferences
                        ).also { migratedValue ->
                            isAvailable = migratedValue
                        }
                        SharedPreferencesUtils.readLong(
                            CONFIG_KEY_THRESHOLD,
                            TERMINATIONS_THRESHOLD.second,
                            crashesPreferences
                        ).also { migratedValue ->
                            threshold = migratedValue
                        }
                        SharedPreferencesUtils.readFloat(
                            CONFIG_KEY_STATE_RATIO,
                            TERMINATIONS_STATE_RATIO.second,
                            crashesPreferences
                        ).also { migratedValue ->
                            stateRatio = migratedValue
                        }
                        setAppTerminationsMigrated()
                    }
                }
            }
    }

    private fun isAppTerminationsMigrated(): Boolean {
        val (key, defaultValue) = Constants.Preferences.IS_TERMINATIONS_MIGRATED
        return SharedPreferencesUtils.readBoolean(key, defaultValue, crashesPreferences)
    }

    private fun setAppTerminationsMigrated() {
        SharedPreferencesUtils.writeBoolean(
            Constants.Preferences.IS_TERMINATIONS_MIGRATED.first,
            true,
            crashesPreferences
        )
    }

}