package com.instabug.fatalhangs

import android.content.Context
import com.instabug.commons.PluginDelegate
import com.instabug.commons.configurations.ConfigurationsHandler
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.logging.logVerbose
import com.instabug.fatalhangs.configuration.FatalHangsConfigurationProvider
import com.instabug.fatalhangs.di.FatalHangsServiceLocator
import com.instabug.fatalhangs.model.FatalHang
import com.instabug.library.Instabug
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.Features
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.threading.PoolProvider
import com.instabug.library.visualusersteps.ReproCapturingProxy

class FatalHangsPluginDelegate : PluginDelegate {

    private var fatalHangDetectorThread: Thread? = null

    private val configurationHandler: ConfigurationsHandler
            by lazy { FatalHangsServiceLocator.fatalHangsConfigurationHandler }

    private val configurationProvider: FatalHangsConfigurationProvider
            by lazy { FatalHangsServiceLocator.fatalHangsConfigurationProvider }

    private val onFatalHangDetectedCallback: (FatalHang) -> Unit = { fatalHang ->
        val fatalHangsCacheManager = FatalHangsServiceLocator.getFatalHangsCacheManager()
        fatalHangsCacheManager.insert(fatalHang, Instabug.getApplicationContext())
        syncFatalHangs()
    }

    private val reproProxy: ReproCapturingProxy by lazy {
        CommonsLocator.reproProxy
    }

    override fun init(context: Context) {
        configurationHandler.migrateCurrentConfiguration()
        reproProxy.evaluate(configurationProvider)
    }

    override fun start(context: Context) {
        PoolProvider.postIOTaskWithCheck {
            SettingsManager.getInstance()
                ?.reproConfigurations
                ?.let {
                    handleReproStateConfigurations(it.modesMap)
                }
            syncFatalHangs()
        }
    }

    override fun wake() {
        startFatalHangsDetectionIfPossible()
    }

    override fun sleep() {
        stopFatalHangsDetection()
    }

    override fun stop() {
        stopFatalHangsDetection()
    }

    override fun handleSDKCoreEvent(sdkCoreEvent: IBGSdkCoreEvent) {
        when (sdkCoreEvent) {
            IBGSdkCoreEvent.NetworkActivated -> {
                "Fatal hangs received network activated event".logVerbose()
                handleNetworkActivated()
            }

            is IBGSdkCoreEvent.ReproState -> handleReproStateConfigurations(sdkCoreEvent.modesMap)

            is IBGSdkCoreEvent.FeaturesFetched -> handleFeaturesFetched(sdkCoreEvent.response)
            is Features -> handleFatalHangsStateChanged()
            else -> {}
        }
    }

    private fun handleNetworkActivated() {
        if (configurationProvider.isFatalHangsEnabled())
            syncFatalHangs()
    }

    private fun handleFeaturesFetched(featuresResponse: String) {
        configurationHandler.handleConfiguration(featuresResponse)
        handleFatalHangsStateChanged()
    }

    private fun handleFatalHangsStateChanged() {
        reproProxy.evaluate(configurationProvider)
        if (configurationProvider.isFatalHangsEnabled()) {
            startFatalHangsDetectionIfPossible()
        } else {
            stopFatalHangsDetection()
            clearCachedFatalHangs()
        }
    }

    private fun hasStartedActivities(): Boolean {
        return InstabugCore.getStartedActivitiesCount() > 0
    }

    private fun startFatalHangsDetectionIfPossible() {
        if (configurationProvider.isFatalHangsEnabled() && fatalHangDetectorThread == null && hasStartedActivities()) {
            fatalHangDetectorThread =
                FatalHangsServiceLocator.getFatalHangDetectorThread(onFatalHangDetectedCallback)
                    .apply { start() }
        }
    }

    private fun syncFatalHangs() {
        synchronized(this::class.java.name) {
            FatalHangsServiceLocator.getFatalHangsSyncManager().syncFatalHangs()
        }
    }

    private fun stopFatalHangsDetection() {
        fatalHangDetectorThread?.interrupt()
        fatalHangDetectorThread = null
    }

    private fun clearCachedFatalHangs() {
        FatalHangsServiceLocator.getIOExecutor()?.execute {
            FatalHangsServiceLocator.getFatalHangsCacheManager()
                .deleteAll(FatalHangsServiceLocator.getContext())
        }
    }

    private fun handleReproStateConfigurations(modesMap: Map<Int, Int>) {
        configurationHandler.handle(modesMap)
        reproProxy.evaluate(configurationProvider)
    }
}