package com.instabug.early_crash.network

import com.instabug.commons.logging.logError
import com.instabug.commons.logging.runOrReportError
import com.instabug.crash.Constants
import com.instabug.crash.OnCrashSentCallback
import com.instabug.crash.models.CrashMetadata
import com.instabug.early_crash.caching.IEarlyCrashCacheHandler
import com.instabug.early_crash.threading.ExecutionMode
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.map.Mapper
import com.instabug.library.networkv2.RequestResponse
import org.json.JSONObject
import java.util.concurrent.ExecutionException
import java.util.concurrent.Executor
import java.util.concurrent.ExecutorService
import java.util.concurrent.TimeUnit

fun interface IEarlyCrashUploaderJob {
    operator fun invoke()
}

class NormalEarlyCrashUploaderJob(
    private val cacheHandler: IEarlyCrashCacheHandler,
    private val uploader: SingleEarlyCrashUploader<Runnable?>,
    private val executor: Executor,
    private val metadataMapper: Mapper<Pair<JSONObject, RequestResponse?>, CrashMetadata?>,
    private val crashMetadataCallback: OnCrashSentCallback
) : IEarlyCrashUploaderJob {

    override operator fun invoke() = executor.execute {
        runCatching {
            cacheHandler.getIds(ExecutionMode.OrderedCaching).forEach { id ->
                syncCrashWithId(id)
            }
        }.runOrReportError("Something went wrong while retrieving crashes", true)
    }

    private fun syncCrashWithId(id: String) = runCatching {
        cacheHandler.get(id, ExecutionMode.OrderedCaching)?.also { jsonObject ->
            uploader(id, jsonObject, ExecutionMode.OrderedCaching) {
                notifyCrashUploaded(jsonObject, it)
            }?.run()
        } ?: "Something went wrong retrieving crash with id $id".logError()
    }.runOrReportError("Something went wrong retrieving crash with id $id", true)

    private fun notifyCrashUploaded(jsonObject: JSONObject, response: RequestResponse?) =
        metadataMapper.map(jsonObject to response)?.let {
            crashMetadataCallback.onCrashSent(it)
        }

}

class AppStartupEarlyCrashUploaderJob(
    private val cacheHandler: IEarlyCrashCacheHandler,
    private val uploader: SingleEarlyCrashUploader<Runnable?>,
    private val executor: ExecutorService,
    private val timeoutSeconds: Long
) : IEarlyCrashUploaderJob {
    override operator fun invoke() {
        runCatching {
            executor.submit {
                cacheHandler.getMostRecent()
                    ?.let { uploader(it.first, it.second, ExecutionMode.Immediate, null) }
                    ?.run()
                Unit
            }.get(timeoutSeconds, TimeUnit.SECONDS)
        }.onFailure { exception ->
            when (exception) {
                is ExecutionException ->
                    IBGDiagnostics.reportNonFatalAndLog(
                        exception,
                        "Failed to sync most recent early crash",
                        Constants.LOG_TAG
                    )

                else -> "Failed to sync most recent early crash".logError(exception)
            }
        }
    }

}

