package com.instabug.crash

import android.content.Context
import com.instabug.commons.IUncaughtExceptionHandlingDelegate
import com.instabug.commons.logging.logDebug
import com.instabug.commons.logging.logError
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.threading.CrashDetailsParser
import com.instabug.crash.ReportCreationHelper.getReport
import com.instabug.crash.ReportCreationHelper.handleReportAttachments
import com.instabug.crash.ReportCreationHelper.modifyReportMetadataStateWithUserInputs
import com.instabug.crash.ReportCreationHelper.modifyReportStateLogsAndAttachmentsWithUserInputs
import com.instabug.crash.ReportCreationHelper.performPostReportActivities
import com.instabug.crash.ReportCreationHelper.performPreReportActivities
import com.instabug.crash.ReportCreationHelper.prepareMetadataState
import com.instabug.crash.ReportCreationHelper.rebuildStateWithStateLogs
import com.instabug.crash.ReportCreationHelper.updateCrash
import com.instabug.crash.di.CrashesServiceLocator
import com.instabug.crash.models.Crash
import com.instabug.crash.network.CrashMetadataImmediateUploader
import com.instabug.crash.utils.addReproScreenshotsAttachmentIfApplicable
import com.instabug.crash.utils.updateReproInteractionsIfApplicable
import com.instabug.library.diagnostics.IBGDiagnostics
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit

class ImmediateSyncUncaughtExceptionHandlerDelegate(
    private val uploader: CrashMetadataImmediateUploader?,
    private val uploadTimeOutSeconds: Long
): IUncaughtExceptionHandlingDelegate {

    override fun invoke(parser: CrashDetailsParser, ctx: Context?) {
        ctx?.let { context ->
            val state = prepareMetadataState(context)
            val report = getReport()
            modifyReportMetadataStateWithUserInputs(state, report)
            var crash: Crash? = Crash.Factory()
                .create(state, context, false, false)
            crash = updateCrash(crash!!, parser)
            val uploaderRunnableFuture = uploader?.invoke(crash)

            performPreReportActivities()
            rebuildStateWithStateLogs(state, context)
            modifyReportStateLogsAndAttachmentsWithUserInputs(state, report)
            crash.updateReproInteractionsIfApplicable()
            crash.addReproScreenshotsAttachmentIfApplicable(context)
            handleReportAttachments(crash, context)
            InstabugUncaughtExceptionHandler.cacheCrash(context, crash)
            performPostReportActivities(crash)
            "Crash report created".logDebug()
            runCatching {
                uploaderRunnableFuture?.get(uploadTimeOutSeconds, TimeUnit.SECONDS)
                    ?.let {
                        it.run()
                        "Crash metadata synced".logVerbose()
                    }
            }.onFailure {
                if (it is ExecutionException) {
                    IBGDiagnostics.reportNonFatalAndLog(
                        it,
                        "Error while performing immediate crash upload",
                        Constants.LOG_TAG)
                } else {
                    "Error while performing immediate crash upload".logError(it)
                }
            }
        }
    }

    class Factory {
        fun create(): IUncaughtExceptionHandlingDelegate =
            ImmediateSyncUncaughtExceptionHandlerDelegate(
                CrashesServiceLocator.immediateCrashMetadataUploader,
                3
            )
    }
}