package com.instabug.commons.session

import androidx.annotation.WorkerThread
import com.instabug.commons.di.CommonsLocator
import com.instabug.commons.logging.logVerbose
import com.instabug.commons.models.Incident
import com.instabug.commons.models.Incident.Type
import com.instabug.commons.session.SessionIncident.ValidationStatus
import com.instabug.library.core.InstabugCore
import com.instabug.library.sessionV3.configurations.IBGSessionCrashesConfigurations
import com.instabug.library.sessionV3.sync.NoneFilter

/**
 * Links incidents to v3 sessions
 */
interface SessionLinker {
    /**
     * Links and incident of type [Incident] with currently running session.
     * @param incident The [Incident] to be linked to the currently running session.
     * @param validationStatus The [ValidationStatus] of the incident.
     */
    @WorkerThread
    fun link(
        incident: Incident,
        @ValidationStatus validationStatus: Int
    )

    @WorkerThread
    fun weakLink(sessionId: String, type: Type)

    /**
     * Validates the session-incident link
     * @param incident the [Incident] that should be validated
     */
    @WorkerThread
    fun validate(incident: Incident)

    @WorkerThread
    fun validateWeakLink(sessionId: String?, incidentId: String?, incidentType: Type)

    @WorkerThread
    fun cleanseByType(incidentType: Type)

    @WorkerThread
    fun inspectDataReadiness(sessionIds: List<String>): Map<String, Boolean>
}

class DefaultSessionLinker : SessionLinker {

    private val cachingHandler: SessionIncidentCachingHandler
        get() = CommonsLocator.sessionIncidentCachingHandler

    private val configurations: IBGSessionCrashesConfigurations
        get() = CommonsLocator.sessionCrashesConfigurations

    override fun link(incident: Incident, @ValidationStatus validationStatus: Int) {
        val incidentId = incident.metadata.uuid ?: run {
            "Session-Incident linking failed, incident doesn't have uuid".logVerbose()
            return
        }
        val sessionId = InstabugCore.getLatestV3SessionId() ?: run {
            "Session-Incident linking failed, v3 session is not available".logVerbose()
            return
        }
        SessionIncident(sessionId, incidentId, incident.type, validationStatus)
            .also(cachingHandler::store)
            .also(this::executePostStoreActions)
    }

    override fun weakLink(sessionId: String, type: Type) {
        val v3SessionId = InstabugCore.getLatestV3SessionId() ?: run {
            "Session-Incident linking failed, v3 session is not available".logVerbose()
            return
        }
        if (sessionId != v3SessionId) {
            "Session id provided for weak link doesn't match latest v3 session id, aborting ..".logVerbose()
            return
        }
        SessionIncident(v3SessionId, null, type, ValidationStatus.UNVALIDATED)
            .also(cachingHandler::store)
            .also(this::executePostStoreActions)
        "Trm weak link created for session $sessionId".logVerbose()
    }

    override fun validate(incident: Incident) {
        val incidentId = incident.metadata.uuid ?: run {
            "Session-Incident validation failed, incident doesn't have uuid".logVerbose()
            return
        }
        cachingHandler.updateValidationStatus(incidentId, ValidationStatus.VALIDATED)
        InstabugCore.notifyV3SessionDataReadiness(NoneFilter)
    }

    override fun validateWeakLink(sessionId: String?, incidentId: String?, incidentType: Type) {
        sessionId ?: run {
            "Session-Incident linking failed, v3 session is not available".logVerbose()
            return
        }
        cachingHandler.updateValidationStatus(
            sessionId,
            incidentId,
            incidentType,
            ValidationStatus.VALIDATED
        )
    }

    override fun cleanseByType(incidentType: Type) {
        cachingHandler.cleanseByIncidentType(incidentType)
    }

    override fun inspectDataReadiness(sessionIds: List<String>): Map<String, Boolean> =
        cachingHandler.queryIncidentsBySessionsIds(sessionIds)
            .groupBy { it.sessionId }
            .mapValues { it.value.fold(true) { acc, sessionIncident -> acc && sessionIncident.isValidated() } }
            .toMutableMap()
            .apply { putAll((sessionIds - keys).associateWith { true }) }

    private fun executePostStoreActions(sessionIncident: SessionIncident) =
        when (val incidentType = sessionIncident.incidentType) {
            Type.NonFatalCrash -> cachingHandler.trim(
                sessionIncident.sessionId,
                incidentType,
                configurations.nonFatalStoreLimit
            )
            Type.ANR -> cachingHandler.trim(
                sessionIncident.sessionId,
                incidentType,
                configurations.anrStoreLimit
            )
            Type.FatalHang -> cachingHandler.trim(
                sessionIncident.sessionId,
                incidentType,
                configurations.fatalHangStoreLimit
            )
            else -> Unit
        }
}
