package com.instabug.anr;

import static com.instabug.commons.session.SessionIncident.ValidationStatus;
import static com.instabug.library.util.threading.DefensiveRunnableKt.runDefensive;

import android.app.ActivityManager;
import android.os.Debug;

import com.instabug.anr.di.AnrServiceLocator;
import com.instabug.anr.diagnostics.ANRIncidentType;
import com.instabug.anr.model.Anr;
import com.instabug.commons.Detection;
import com.instabug.commons.di.CommonsLocator;
import com.instabug.commons.diagnostics.di.DiagnosticsLocator;
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent;
import com.instabug.commons.diagnostics.event.CalibrationDiagnosticEvent.Action;
import com.instabug.commons.models.IncidentMetadata;
import com.instabug.crash.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.io.IOException;

/**
 * A thread class intended to detect ANRs
 * Here is how it works:
 * 1. Post a runnable to the main thread after {@value CHECK_INTERVAL}
 * 2. Sleep the thread for the {@value CHECK_INTERVAL}
 * 3. Check if the posted runnable was run or not:
 * if not run, then check if the passed milliseconds (represented by: count) exceeds the {@value CHECK_INTERVAL}:
 * if count exceeds: then ANR was detected
 */
public class InstabugAnrDetectorThread extends Thread {

    private final static long CHECK_INTERVAL = 500L;
    // Initial value with true to post the first Runnable to the main thread
    private boolean isAnrReported = false;
    private boolean isInterrupted = false;
    private AnrListener anrListener;
    private Anr.Factory anrFactory;
    private ProcessStateHelper processStateHelper;

    public InstabugAnrDetectorThread(AnrListener anrListener, Anr.Factory anrFactory, ProcessStateHelper processStateHelper) {
        this.anrListener = anrListener;
        this.anrFactory = anrFactory;
        this.processStateHelper = processStateHelper;
    }

    @Override
    public void interrupt() {
        super.interrupt();
        isInterrupted = true;
    }

    @Override
    public void run() {
        setName("Instabug ANR detector thread");
        runDefensive(() -> {
            // adding a flag isInterrupted because isInterrupted() method is not breaking the while loop in case of interruption.
            while (Instabug.isEnabled() && !isInterrupted() && !isInterrupted) {
                if (!Debug.isDebuggerConnected() && !Debug.waitingForDebugger()) {
                    ActivityManager.ProcessErrorStateInfo processState = processStateHelper.getProcessErrorState();
                    if (!isAnrReported && anrListener != null) {
                        if (processState != null && processState.condition == ActivityManager.ProcessErrorStateInfo.NOT_RESPONDING) {
                            try {
                                DiagnosticsLocator.getReporter().report(new CalibrationDiagnosticEvent(new ANRIncidentType(), Action.Captured));
                                AnrServiceLocator.getAnrDetectorListener().onDetection(Detection.Anr);
                                Anr anr = anrFactory.createAnr(processState.shortMsg, processStateHelper.getLongMessageForAnr(processState), IncidentMetadata.Factory.create());
                                if (anr != null) {
                                    CommonsLocator.getSessionLinker().link(anr, ValidationStatus.VALIDATED);
                                    anrListener.onAnrDetected(anr);
                                }
                            } catch (JSONException e) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't create a new ANR object " +
                                        "due to a JSON exception", e);

                            } catch (IOException ioE) {
                                InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't create a new ANR object " +
                                        "due to an IO exception", ioE);
                            }

                            isAnrReported = true;
                        }
                    } else if (processState == null) {
                        if (isAnrReported) {
                            AnrServiceLocator.getAnrDetectorListener().onDetection(Detection.AnrRecovery);
                        }
                        isAnrReported = false;
                    }
                }

                try {
                    sleep(CHECK_INTERVAL);
                } catch (InterruptedException e) {
                   //to do add suitable log
                }
            }
        }).run();
    }
}
