package com.instabug.library.util.threading

import androidx.annotation.VisibleForTesting

/**
 * A custom implementation of the NetworkingSingleThreadPoolExecutor class that can control
 * the commands (runnables) and execute them in order (one after another) or cancel them if
 * required (If the thread is sleeping and they are not currently running)
 * and execute the new incoming command.
 */
class RetryableNetworkingSingleThreadExecutor(
    identifier: String
) : NetworkingSingleThreadPoolExecutor(identifier) {

    @VisibleForTesting
    internal var currentNetworkingThread: Thread? = null

    @Synchronized
    override fun execute(command: Runnable?) {
        if (currentNetworkingThread?.state == Thread.State.TIMED_WAITING) {
            queue.clear()
            currentNetworkingThread?.interrupt()
        }

        super.execute(wrapCommand(command))
    }

    /**
     * Wrap the incoming command in another runnable to be able to get a reference to the thread
     * that executes it to check if it's sleeping or running when a new command is posted later on.
     */
    private fun wrapCommand(command: Runnable?): Runnable {
        return Runnable {
            currentNetworkingThread = Thread.currentThread()
            command?.run()
            currentNetworkingThread = null
        }
    }

}