package com.instabug.library.util;


import static com.instabug.library.internal.storage.DiskUtils.copyFromUriIntoFile;
import static com.instabug.library.internal.storage.DiskUtils.deleteFile;
import static com.instabug.library.util.FileUtils.FileType.FOLDER;
import static com.instabug.library.util.FileUtils.FileType.IMAGE;
import static com.instabug.library.util.FileUtils.FileType.UNKNOWN;
import static com.instabug.library.util.FileUtils.FileType.VIDEO;
import static java.lang.annotation.RetentionPolicy.SOURCE;
import static javax.crypto.Cipher.DECRYPT_MODE;
import static javax.crypto.Cipher.ENCRYPT_MODE;

import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.webkit.URLUtil;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.internal.storage.ProcessedBytes;
import com.instabug.library.util.memory.MemoryUtils;
import com.instabug.library.util.threading.PoolProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.annotation.Retention;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.crypto.Cipher;

public final class FileUtils {

    private static final int NOT_FOUND = -1;
    private static final char EXTENSION_SEPARATOR = '.';
    private static final char UNIX_SEPARATOR = '/';
    public static final String FLAG_ENCRYPTED = "_e";
    private static final int NUMBER_BYTES_TO_PROCESS = 256;
    private static final int IV_LENGTH;

    static {
        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            IV_LENGTH = 12;
        } else {
            IV_LENGTH = 16;
        }
    }

    /**
     * Extracts extension from file and detects its type
     *
     * @param file the file to get its type
     * @return file type as represented in {@link FileType}
     */
    @FileType
    public static int getFileType(@NonNull File file) {
        if (file.isDirectory()) {
            return FOLDER;
        }
        String absolutePath = file.getAbsolutePath();
        int index = getIndexOfExtension(absolutePath);
        return index == NOT_FOUND ? UNKNOWN : getTypeFromExtension(absolutePath);
    }

    /**
     * Detects whether or not the passed argument is local path or network url and return the
     * appropriate intent with action view
     *
     * @param pathOrUrl the local path or network url
     * @return the appropriate intent with action {@link Intent#ACTION_VIEW}
     */
    public static Intent getFileViewerIntent(@NonNull String pathOrUrl) {
        if (URLUtil.isNetworkUrl(pathOrUrl)) {
            return new Intent(Intent.ACTION_VIEW, Uri.parse(pathOrUrl));
        } else {
            String extension = getExtension(pathOrUrl);
            Uri uri = Uri.fromFile(new File(pathOrUrl));
            return appropriateIntent(uri, extension);
        }
    }

    /**
     * Extracts extension from file and returns it as string
     *
     * @param file the file to extract the extension from
     * @return the extension or empty string if non was found
     */
    public static String getExtension(@NonNull File file) {
        return getExtension(file.getAbsolutePath());
    }

    /**
     * Extracts extension from absolutePath and returns it as string
     *
     * @param absolutePath the absolute file path name to extract the extension from
     * @return the extension or empty string if non was found
     */
    public static String getExtension(@NonNull String absolutePath) {
        int index = getIndexOfExtension(absolutePath);
        return index == NOT_FOUND ? "" : absolutePath.substring(index + 1);
    }

    public static boolean isVideoFile(@NonNull File file) {
        if (!file.isFile()) {
            return false;
        }
        String extension = getExtension(file);
        return isVideoExtension(extension);
    }

    public static boolean isImageFile(@NonNull File file) {
        if (!file.isFile()) {
            return false;
        }
        String extension = getExtension(file);
        return isImageExtension(extension);
    }

    public static boolean isTextFile(@NonNull File file) {
        if (!file.isFile()) {
            return false;
        }
        String extension = getExtension(file.getAbsolutePath());
        return isTextExtension(extension);
    }

    public static boolean isCacheFile(@NonNull File file) {
        if (!file.isFile()) {
            return false;
        }
        String extension = getExtension(file.getAbsolutePath());
        return isCacheExtension(extension);
    }

    public static boolean isVideoExtension(@NonNull String extension) {
        return extension.equalsIgnoreCase("mp4")
                || extension.equalsIgnoreCase("avi")
                || extension.equalsIgnoreCase("mpg")
                || extension.equalsIgnoreCase("3gp")
                || extension.equalsIgnoreCase("3gpp")
                || extension.equalsIgnoreCase("ts")
                || extension.equalsIgnoreCase("AAC")
                || extension.equalsIgnoreCase("webm")
                || extension.equalsIgnoreCase("mkv");
    }

    public static boolean isImageExtension(@NonNull String extension) {
        return extension.equalsIgnoreCase("jpeg")
                || extension.equalsIgnoreCase("gif")
                || extension.equalsIgnoreCase("png")
                || extension.equalsIgnoreCase("bmp")
                || extension.equalsIgnoreCase("jpg")
                || extension.equalsIgnoreCase("webp");
    }

    private static boolean isTextExtension(@NonNull String extension) {
        return extension.equalsIgnoreCase("txt");
    }

    private static boolean isCacheExtension(@NonNull String extension) {
        return extension.equalsIgnoreCase("cache");
    }

    public static void sortByLastModifiedAsc(@NonNull List<File> files) {
        try {
            Collections.sort(files, new Comparator<File>() {
                @Override
                public int compare(File f1, File f2) {
                    return Double.compare(f1.lastModified(), f2.lastModified());
                }
            });
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Exception " + e.getMessage() + " while sorting list");
        }
    }

    public static long getSize(@NonNull File file) {
        if (!file.exists()) return 0;
        long size = file.length();
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files != null) {
                for (File value : files) {
                    size += getSize(value);
                }
            }
        }
        return size;
    }

    @FileType
    private static int getTypeFromExtension(String absolutePath) {
        String extension = getExtension(absolutePath);
        if (isVideoExtension(extension)) {
            return VIDEO;
        } else if (isImageExtension(extension)) {
            return IMAGE;
        } else {
            return UNKNOWN;
        }
    }

    private static Intent appropriateIntent(Uri uri, String extension) {
        Intent intent = new Intent(Intent.ACTION_VIEW);
        if (extension.equalsIgnoreCase("doc") || extension.equalsIgnoreCase("docx")) {
            // Word document
            intent.setDataAndType(uri, "application/msword");
        } else if (extension.equalsIgnoreCase("pdf")) {
            // PDF file
            intent.setDataAndType(uri, "application/pdf");
        } else if (extension.equalsIgnoreCase("ppt") || extension.equalsIgnoreCase("pptx")) {
            // Powerpoint file
            intent.setDataAndType(uri, "application/vnd.ms-powerpoint");
        } else if (extension.equalsIgnoreCase("xls") || extension.equalsIgnoreCase("xlsx")) {
            // Excel file
            intent.setDataAndType(uri, "application/vnd.ms-excel");
        } else if (extension.equalsIgnoreCase("zip") || extension.equalsIgnoreCase("rar")) {
            // WAV audio file
            intent.setDataAndType(uri, "application/x-wav");
        } else if (extension.equalsIgnoreCase("rtf")) {
            // RTF file
            intent.setDataAndType(uri, "application/rtf");
        } else if (extension.equalsIgnoreCase("wav") || extension.equalsIgnoreCase("mp3")) {
            // WAV audio file
            intent.setDataAndType(uri, "audio/x-wav");
        } else if (extension.equalsIgnoreCase("gif")) {
            // GIF file
            intent.setDataAndType(uri, "image/gif");
        } else if (extension.equalsIgnoreCase("jpg") || extension.equalsIgnoreCase("jpeg") || extension.equalsIgnoreCase("png")) {
            // JPG file
            intent.setDataAndType(uri, "image/jpeg");
        } else if (extension.equalsIgnoreCase("txt")) {
            // Text file
            intent.setDataAndType(uri, "text/plain");
        } else if (extension.equalsIgnoreCase("3gp") || extension.equalsIgnoreCase("mpg") || extension.equalsIgnoreCase("mpeg") ||
                extension.equalsIgnoreCase("mpe") || extension.equalsIgnoreCase("mp4") || extension.equalsIgnoreCase("avi")) {
            // Video files
            intent.setDataAndType(uri, "video/*");
        } else {
            intent.setDataAndType(uri, "*/*");
        }
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        return intent;
    }

    @Nullable
    public static File getFile(String path) {
        File file = new File(path);
        if (file.exists())
            return file;
        else return null;
    }

    /*
     * This method Returns true if the passed filePath is related to a Bug or a Crash report
     * It is used by InstabugDelegate.class to determine whether or not we should delete this file
     * due to the latest changes in Encryptor
     * */
    public static boolean isFileRelatedToBugOrCrashReport(String filePath) {
        return
                // User steps folder
                filePath.contains("vusf")
                        // View hierarchy images
                        || filePath.contains("view-hierarchy-images")
                        // Bug screenshots
                        || (filePath.contains("bug_") && filePath.endsWith("_.jpg"))
                        // View hierarchy zip file
                        || (filePath.contains("view_hierarchy_attachment_") && filePath.endsWith(".zip"))
                        // Repro steps zip file
                        || (filePath.contains("usersteps_") && filePath.endsWith(".zip"));
    }

    public static int getIndexOfExtension(String filename) {
        int extensionPos = filename.lastIndexOf(EXTENSION_SEPARATOR);
        int lastSeparator = filename.lastIndexOf(UNIX_SEPARATOR);
        return lastSeparator > extensionPos ? NOT_FOUND : extensionPos;
    }

    public static boolean isEncryptedFile(final String filePath) {
        Boolean result = PoolProvider.getFilesEncryptionExecutor().executeAndGet(new ReturnableRunnable<Boolean>() {
            @Nullable
            @Override
            public Boolean run() {
                int extensionIndex = getIndexOfExtension(filePath);
                String pathWithoutExtension = filePath.substring(0, extensionIndex);
                return pathWithoutExtension.endsWith(FLAG_ENCRYPTED);
            }
        });
        return result != null ? result : false;
    }

    public static String getPathWithEncryptedFlag(String path) {
        int extensionIndex = FileUtils.getIndexOfExtension(path);
        if (extensionIndex != -1) {
            String pathWithoutExtension = path.substring(0, extensionIndex);
            String extension = path.substring(extensionIndex);
            return String.format("%s%s%s", pathWithoutExtension, FLAG_ENCRYPTED, extension);
        }
        return "";
    }

    public static String getPathWithDecryptedFlag(String path) {
        return path.replace(FLAG_ENCRYPTED, "");
    }

    public static boolean isReproStepFile(String filePath) {
        return (filePath.contains("step") || filePath.contains("icon")) && filePath.endsWith(".png") &&
                !filePath.contains("usersteps_") && !filePath.endsWith(".zip");
    }

    public static boolean encryptFile(@NonNull final String path) throws UnsatisfiedLinkError {
        Boolean result = PoolProvider.getFilesEncryptionExecutor().executeAndGet(new ReturnableRunnable<Boolean>() {
            @Nullable
            @Override
            public Boolean run() {
                File file = new File(path);
                boolean isFileEncrypted = fileProcessor(ENCRYPT_MODE, file);
                if (isFileEncrypted && (isReproStepFile(path) || isInternalAttachmentFile(path))) {
                    String newPath = getPathWithEncryptedFlag(path);
                    if (!newPath.equals("")) {
                        file.renameTo(new File(newPath));
                    }
                }
                return isFileEncrypted;
            }
        });
        return result != null ? result : false;
    }

    private static boolean isInternalAttachmentFile(@NonNull String path) {
        return path.contains("internal-attachments");
    }

    public static boolean decryptFile(@NonNull final String filePath) throws UnsatisfiedLinkError {
        Boolean result = PoolProvider.getFilesEncryptionExecutor().executeAndGet(new ReturnableRunnable<Boolean>() {
            @Nullable
            @Override
            public Boolean run() {
                File file = new File(filePath);
                boolean isFileDecrypted = fileProcessor(DECRYPT_MODE, file);
                if (isFileDecrypted && (isReproStepFile(filePath) || isInternalAttachmentFile(filePath))) {
                    String newPath = getPathWithDecryptedFlag(filePath);
                    file.renameTo(new File(newPath));
                }
                return isFileDecrypted;
            }
        });
        return result != null ? result : false;
    }

    private static boolean fileProcessor(int cipherMode, File file) {
        FileInputStream fis = null;
        RandomAccessFile raf = null;
        Context context = Instabug.getApplicationContext();
        if (context != null && !MemoryUtils.isLowMemory(context)) {
            String cipherModeDescription = "";
            if (cipherMode == ENCRYPT_MODE) {
                cipherModeDescription = "encrypting";
            } else if (cipherMode == DECRYPT_MODE) {
                cipherModeDescription = "decrypting";
            }
            try {
                // Get the first NUMBER_BYTES_TO_PROCESS bytes and process them
                fis = new FileInputStream(file);
                raf = new RandomAccessFile(file, "rws");

                byte[] bytesToProcess;
                byte[] processedBytes;
                long proccessBytesFromPosition = 0;
                int numOfBytesToProcess = (int) Math.min(file.length(), NUMBER_BYTES_TO_PROCESS);

                if (cipherMode == Cipher.DECRYPT_MODE && file.length() > NUMBER_BYTES_TO_PROCESS) {
                    //Encryption process adds extra bytes [IV_LENGTH] so consider them in the opposite process.
                    //Add IV_LENGTH to number of bytes to process in case of decryption
                    numOfBytesToProcess += IV_LENGTH;
                }

                if (file.length() > numOfBytesToProcess) {
                    //Process only the last (n) bytes in case of a large file and to avoid overriding
                    // some bytes if the output of the encryption is greater than the original bytes length.
                    proccessBytesFromPosition = file.length() - numOfBytesToProcess;
                }

                bytesToProcess = new byte[numOfBytesToProcess];

                raf.seek(proccessBytesFromPosition);
                raf.read(bytesToProcess, 0, bytesToProcess.length);

                if (cipherMode == Cipher.ENCRYPT_MODE) {
                    processedBytes = EncryptionManager.encrypt(bytesToProcess);
                } else {
                    processedBytes = EncryptionManager.decrypt(bytesToProcess);
                }

                //Remove the last (n) bytes [bytesToProcess] from the file and write the generated ones [processedBytes].
                raf.setLength(raf.length() - numOfBytesToProcess);
                raf.write(processedBytes, 0, processedBytes.length);

                return true;
            } catch (OutOfMemoryError | Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG,
                        String.format("Error: %s occurred while %s file in path: %s", e, cipherModeDescription, file.getPath()));
                IBGDiagnostics.reportNonFatal(e, String.format("Error: %s occurred while %s file in path: %s", e, cipherModeDescription, file.getPath()));
            } finally {
                if (fis != null) {
                    try {
                        fis.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
                if (raf != null) {
                    try {
                        raf.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return false;
    }

    public static ProcessedBytes decryptOnTheFly(String filePath) throws UnsatisfiedLinkError {
        File file = new File(filePath);
        return fileDecryptionOnTheFlyProcessor(file);
    }

    private static ProcessedBytes fileDecryptionOnTheFlyProcessor(File file) {
        RandomAccessFile raf = null;
        try {
            // Get the first NUMBER_BYTES_TO_PROCESS bytes and decrypt them
            raf = new RandomAccessFile(file, "rws");
            byte[] bytesToEncrypt = new byte[NUMBER_BYTES_TO_PROCESS + IV_LENGTH];

            if (raf.length() > (NUMBER_BYTES_TO_PROCESS + IV_LENGTH)) {
                raf.seek(raf.length() - (NUMBER_BYTES_TO_PROCESS + IV_LENGTH));
            }
            raf.read(bytesToEncrypt, 0, bytesToEncrypt.length);

            byte[] decryptedBytes = EncryptionManager.decrypt(bytesToEncrypt);

            // Override the file with the decrypted bytes
            // Mode "rws" according to docs is used for reading/writing file synchronously
            if (raf.length() > (NUMBER_BYTES_TO_PROCESS + IV_LENGTH)) {
                raf.seek(raf.length() - (NUMBER_BYTES_TO_PROCESS + IV_LENGTH));
            }
            raf.write(decryptedBytes, 0, decryptedBytes.length);

            // Read the whole file
            byte[] decryptedFileBytes = new byte[(int) file.length()];
            read(file, decryptedFileBytes);

            // Rename the file to the unencrypted
            if (isReproStepFile(file.getPath())) {
                String newPath = getPathWithDecryptedFlag(file.getPath());
                file.renameTo(new File(newPath));
            }

            return new ProcessedBytes(decryptedFileBytes, true);
        } catch (Exception | OutOfMemoryError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error: " + e + " occurred while decrypting file in path: " + file.getPath());
            IBGDiagnostics.reportNonFatal(e, "Error: " + e + " occurred while decrypting file in path: " + file.getPath());
        } finally {
            try {
                if (raf != null) {
                    raf.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return new ProcessedBytes(new byte[0], false);
    }

    public static void read(File inputFile, byte[] outputBytes) throws IOException {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(inputFile);
            fis.read(outputBytes);
        } finally {
            if (fis != null) {
                fis.close();
            }
        }

    }

    @WorkerThread
    @NonNull
    public static synchronized List<File> getStateFiles(String prefix) {
        ArrayList<File> list = new ArrayList<>();
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            File rootDirectory = context.getFilesDir();
            File parent = rootDirectory.getParentFile();
            FilenameFilter filter = (dir, name) -> name.startsWith(prefix) && name.endsWith(".txt");
            if (parent != null) {
                File[] stateFiles = parent.listFiles(filter);
                if (stateFiles != null) {
                    list.addAll(Arrays.asList(stateFiles));
                }
            }
        }
        return list;
    }

    @Nullable
    public static String getFileName(@NonNull String path) {
        return Uri.parse(path).getLastPathSegment();
    }

    public static void deleteDirectory(@Nullable File file) {
        if (file != null) {
            if (file.isDirectory()) {
                File[] children = file.listFiles();
                if (children != null)
                    for (File child : children)
                        deleteDirectory(child);
            }

            file.delete();
        }
    }

    /**
     * Copies and deletes the original file
     *
     * @param context          Application context
     * @param originalFilePath original file path to be copied
     * @param newFileDirectory path of the directory to copy the file to
     * @return the absolute path of the copied file
     * @throws IOException in case of failure in creating a copy file
     */
    @Nullable
    public static String copyAndDeleteOriginalFile(@NonNull Context context, @Nullable String originalFilePath, @Nullable String newFileDirectory) {

        try {
            if (originalFilePath == null || newFileDirectory == null) {
                return null;
            }
            File copiedFile = new File(newFileDirectory
                    + Uri.parse(originalFilePath).getLastPathSegment());
            File parent = copiedFile.getParentFile();
            if (parent == null) {
                return originalFilePath;
            }
            if (!parent.exists())
                parent.mkdirs();
            copiedFile.createNewFile();
            copyFromUriIntoFile(context, Uri.fromFile(new File(originalFilePath)), copiedFile);
            deleteFile(originalFilePath);
            return copiedFile.getAbsolutePath();
        } catch (IOException ex) {
            InstabugSDKLogger.w(Constants.LOG_TAG, "Something went wrong while copying the file");
        }
        return originalFilePath;
    }

    @Retention(SOURCE)
    @IntDef({UNKNOWN, VIDEO, IMAGE, FOLDER})
    public @interface FileType {
        int UNKNOWN = -1;
        int VIDEO = 0;
        int IMAGE = 1;
        int FOLDER = 2;
    }
}