package com.instabug.library.ui.custom;

import static com.instabug.library.util.ThemeApplier.applyCtaStyle;
import static com.instabug.library.util.ThemeApplier.applyPrimaryTextStyle;
import static com.instabug.library.util.ThemeApplier.applySecondaryTextStyle;
import android.app.Activity;
import android.content.DialogInterface;
import android.os.Build;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;

import com.instabug.library.R;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.AttrResolver;
import com.instabug.library.util.UiUtils;

/**
 * Created by mzelzoghbi on 5/2/18.
 */
public class InstabugAlertDialog {

    public static class Builder {
        @NonNull
        private final Activity activity;
        @Nullable
        private String title;
        @Nullable
        private String message;
        @Nullable
        private String positiveButtonText;
        @Nullable
        private String negativeButtonText;
        @Nullable
        private DialogInterface.OnClickListener onPositiveClickListener;
        @Nullable
        private DialogInterface.OnClickListener onNegativeClickListener;
        @Nullable
        private String positiveButtonAccessibilityContentDescription;
        @Nullable
        private String negativeButtonAccessibilityContentDescription;
        private boolean isCancellable = true;

        public Builder(@NonNull Activity activity) {
            this.activity = activity;
        }

        public Builder setTitle(@Nullable String title) {
            this.title = title;
            return this;
        }

        public Builder setMessage(@Nullable String message) {
            this.message = message;
            return this;
        }

        public Builder setPositiveButton(@Nullable String positiveButtonText, @Nullable DialogInterface.OnClickListener onPositiveClickListener) {
            this.positiveButtonText = positiveButtonText;
            this.onPositiveClickListener = onPositiveClickListener;
            return this;
        }

        public Builder setNegativeButton(@Nullable String negativeButtonText, @Nullable DialogInterface.OnClickListener onNegativeClickListener) {
            this.negativeButtonText = negativeButtonText;
            this.onNegativeClickListener = onNegativeClickListener;
            return this;
        }

        public Builder setCancellable(boolean cancellable) {
            isCancellable = cancellable;
            return this;
        }

        public Builder setPositiveButtonAccessibilityContentDescription(@Nullable String positiveButtonAccessibilityContentDescription) {
            this.positiveButtonAccessibilityContentDescription = positiveButtonAccessibilityContentDescription;
            return this;
        }

        public Builder setNegativeButtonAccessibilityContentDescription(@Nullable String negativeButtonAccessibilityContentDescription) {
            this.negativeButtonAccessibilityContentDescription = negativeButtonAccessibilityContentDescription;
            return this;
        }

        public AlertDialog show() {
            AlertDialog.Builder builder = new AlertDialog.Builder(activity, R.style.InstabugDialogStyle);
            builder.setTitle(title)
                    .setMessage(message)
                    .setCancelable(isCancellable);

            if (positiveButtonText != null) {
                DialogInterface.OnClickListener onPositiveClicked = onPositiveClickListener == null ?
                        (dialogInterface, i) -> dialogInterface.dismiss()
                        : onPositiveClickListener;
                builder.setPositiveButton(positiveButtonText, onPositiveClicked);
            }
            if (negativeButtonText != null) {
                DialogInterface.OnClickListener onNegativeClicked = onNegativeClickListener == null ?
                        (dialogInterface, i) -> dialogInterface.cancel()
                        : onNegativeClickListener;
                builder.setNegativeButton(negativeButtonText, onNegativeClicked);
            }

            AlertDialog alertDialog = builder.create();
            Window window = alertDialog.getWindow();
            if (InstabugCore.isFullScreen() && window != null) {
                window.setFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE, WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);
            }
            alertDialog.setOnShowListener(dialog -> {
                applyCtaStyle(alertDialog.getButton(DialogInterface.BUTTON_POSITIVE), SettingsManager.getInstance().getIBGTheme());
                applyCtaStyle(alertDialog.getButton(DialogInterface.BUTTON_NEGATIVE), SettingsManager.getInstance().getIBGTheme());


                TextView titleView = alertDialog.findViewById(R.id.alertTitle);
                TextView messageView = alertDialog.findViewById(android.R.id.message);
                View container = alertDialog.findViewById(R.id.parentPanel);
                applyPrimaryTextStyle(titleView, SettingsManager.getInstance().getIBGTheme());
                applySecondaryTextStyle(messageView, SettingsManager.getInstance().getIBGTheme());
                
                //To fix incorrect text alignment in rtl layouts
                if (titleView != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                    titleView.setTextAlignment(View.TEXT_ALIGNMENT_TEXT_START);
                }

                if (container != null) {
                    container.setBackgroundColor(getBackgroundColor(activity));
                }
                if (AccessibilityUtils.isTalkbackEnabled()) {
                    alertDialog.getButton(DialogInterface.BUTTON_POSITIVE).setContentDescription(positiveButtonAccessibilityContentDescription);
                    alertDialog.getButton(DialogInterface.BUTTON_NEGATIVE).setContentDescription(negativeButtonAccessibilityContentDescription);
                    if (titleView != null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                        titleView.setImportantForAccessibility(View.IMPORTANT_FOR_ACCESSIBILITY_YES);
                        titleView.setScreenReaderFocusable(true);
                    }
                }

            });

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                if (!activity.isFinishing() && !activity.isDestroyed())
                    alertDialog.show();
            } else {
                if (!activity.isFinishing()) {
                    alertDialog.show();
                }
            }
          
            if (InstabugCore.isFullScreen() && window != null) {
                UiUtils.showInFullscreen(window);
                window.clearFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE);
            }


            return alertDialog;
        }

        private int getBackgroundColor(Activity activity) {
            IBGTheme theme = SettingsManager.getInstance().getIBGTheme();
            if (theme != null && theme.getBackgroundColor() != -1) {
                return theme.getBackgroundColor();
            } else {
                return AttrResolver.getBackgroundColor(activity);
            }
        }
    }
}
