package com.instabug.library.tracking

import android.app.Activity
import android.content.Context
import android.os.Bundle
import android.view.View
import android.view.ViewParent
import androidx.annotation.VisibleForTesting
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import com.instabug.library.InstabugState
import com.instabug.library.InstabugStateProvider
import com.instabug.library._InstabugActivity
import com.instabug.library.internal.lifecycle.DefaultActivityLifecycleCallbacks
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.model.StepType
import com.instabug.library.model.StepType.DIALOG_FRAGMENT_RESUMED
import com.instabug.library.model.StepType.FRAGMENT_ATTACHED
import com.instabug.library.model.StepType.FRAGMENT_DETACHED
import com.instabug.library.model.StepType.FRAGMENT_PAUSED
import com.instabug.library.model.StepType.FRAGMENT_RESUMED
import com.instabug.library.model.StepType.FRAGMENT_STARTED
import com.instabug.library.model.StepType.FRAGMENT_STOPPED
import com.instabug.library.model.StepType.FRAGMENT_VIEW_CREATED
import com.instabug.library.tracking.FragmentNode.LifecycleStage
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Attached
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Detached
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Paused
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Resumed
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Started
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.Stopped
import com.instabug.library.tracking.FragmentNode.LifecycleStage.Companion.ViewCreated
import com.instabug.library.util.extenstions.runOrLogError
import com.instabug.library.visualusersteps.ReproStepsCaptor

class IBGActivityLifecycleMonitor(private val screensRoot: ScreenChildrenOwner) :
    DefaultActivityLifecycleCallbacks {

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        if (isInstabugActivity(activity)) return
        ActivityNode.Factory.create(activity)
            .also(screensRoot::addChild)
            .also(FragmentsScreenChildrenOwner.Helper::injectChildrenObserver)
    }

    override fun onActivityResumed(activity: Activity) {
        if (isInstabugActivity(activity)) return
        screensRoot.findChild(activity.hashCode())?.activate()
    }

    override fun onActivityPaused(activity: Activity) {
        if (isInstabugActivity(activity)) return
        screensRoot.findChild(activity.hashCode())?.deactivate()
    }

    override fun onActivityDestroyed(activity: Activity) {
        if (isInstabugActivity(activity)) return
        with(screensRoot) {
            (findChild(activity.hashCode()) as? FragmentsScreenChildrenOwner)
                ?.let(FragmentsScreenChildrenOwner.Helper::removeChildrenObserver)
            removeChild(activity.hashCode())
        }
    }

    private fun isInstabugActivity(activity: Activity): Boolean = activity is _InstabugActivity
}

class IBGFragmentLifecycleMonitor(private val parent: FragmentsScreenChildrenOwner) :
    FragmentManager.FragmentLifecycleCallbacks() {

    override fun onFragmentAttached(fm: FragmentManager, f: Fragment, context: Context) {
        super.onFragmentAttached(fm, f, context)
        FragmentNode.Factory.create(f, parent)
            .also(parent::addChild)
            .also(FragmentsScreenChildrenOwner.Helper::injectChildrenObserver)
            .also { node -> FragmentsReproStepsHandler.addReproStep(Attached, node, parent) }
    }

    override fun onFragmentViewCreated(
        fm: FragmentManager,
        f: Fragment,
        v: View,
        savedInstanceState: Bundle?
    ) {
        super.onFragmentViewCreated(fm, f, v, savedInstanceState)
        findFragmentChild(f.hashCode())
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(ViewCreated, node, parent) }
            ?.takeIf { isInstabugInEnabledState() }
            ?.also { CoreServiceLocator.navigableViewsTracker.onFragmentViewCreated(f) }
    }

    override fun onFragmentStarted(fm: FragmentManager, f: Fragment) {
        super.onFragmentStarted(fm, f)
        findFragmentChild(f.hashCode())
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(Started, node, parent) }
    }

    override fun onFragmentResumed(fm: FragmentManager, f: Fragment) {
        findFragmentChild(f.hashCode())
            ?.apply { if (!isActive) activate() }
            ?.takeIf { node -> node.isVisible }
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(Resumed, node, parent) }
    }

    override fun onFragmentPaused(fm: FragmentManager, f: Fragment) {
        findFragmentChild(f.hashCode())
            ?.apply { deactivate() }
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(Paused, node, parent) }
    }

    override fun onFragmentStopped(fm: FragmentManager, f: Fragment) {
        super.onFragmentStopped(fm, f)
        findFragmentChild(f.hashCode())
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(Stopped, node, parent) }
    }

    override fun onFragmentDetached(fm: FragmentManager, f: Fragment) {
        findFragmentChild(f.hashCode())
            ?.also(FragmentsScreenChildrenOwner.Helper::removeChildrenObserver)
            ?.also { node -> FragmentsReproStepsHandler.addReproStep(Detached, node, parent) }
            ?.also { node -> parent.removeChild(node.id) }
            ?.also(FragmentNode::clean)
        CoreServiceLocator.navigableViewsTracker.onFragmentDetached(f)
    }

    private fun findFragmentChild(id: Int): FragmentNode? =
        parent.findChild(id)?.let { child -> child as? FragmentNode }
}

object FragmentsReproStepsHandler {
    fun addReproStep(
        @LifecycleStage stage: Int,
        child: FragmentNode,
        parent: FragmentsScreenChildrenOwner
    ) {
        if (!isInstabugInEnabledState()) return
        if (child.lifecycleStage and stage > 0) return
        child.updateLifecycleStage(toCumulativeStage(stage))
        (parent as? FragmentNode)?.let { node -> addParentReproStep(stage, node) }
        toStepType(stage, child)?.let { step ->
            CoreServiceLocator.reproStepsProxy
                .addVisualUserStep(step, child.simpleName, child.fullName, null)
        }
    }

    fun addParentReproStep(@LifecycleStage stage: Int, parent: FragmentNode) {
        val shouldUpdateParentFirst =
            parent.lifecycleStage and stage == 0 && stage < Paused
        if (!shouldUpdateParentFirst) return
        parent.updateLifecycleStage(toCumulativeStage(stage))
        toStepType(stage, parent)?.let { step ->
            CoreServiceLocator.reproStepsProxy
                .addVisualUserStep(step, parent.simpleName, parent.fullName, null)
        }
    }

    private fun toCumulativeStage(@LifecycleStage stage: Int) = when {
        stage < Paused -> stage.toPreResumeCumulative()
        stage == Paused -> Paused
        else -> stage.toPostPauseCumulative()
    }

    private fun toStepType(@LifecycleStage stage: Int, node: FragmentNode) = when (stage) {
        Attached -> FRAGMENT_ATTACHED
        ViewCreated -> FRAGMENT_VIEW_CREATED
        Started -> FRAGMENT_STARTED
        Resumed -> getResumedStep(node)
        Paused -> FRAGMENT_PAUSED
        Stopped -> FRAGMENT_STOPPED
        Detached -> FRAGMENT_DETACHED
        else -> null
    }

    private fun getResumedStep(node: FragmentNode): String =
        DIALOG_FRAGMENT_RESUMED.takeIf { node.isDialogFragment } ?: FRAGMENT_RESUMED
}

/**
 * Once a stage pre-resume (ex ViewCreated) is reached, by default the stages before it should
 * be passed.
 * for example:
 * ViewCreated => 0010 (binary)
 * When reached, we need to cumulate the previous stages in the final representation.
 * Hence the output would be 0011 (binary) including Attached stage
 * example 2:
 * Stage => Resumed = 1000 (binary)
 * output => 1111 (cumulative binary)
 */
@VisibleForTesting
fun Int.toPreResumeCumulative(): Int = this or trailingMask(this.countTrailingZeroBits())

/**
 * Once a stage post-pause (ex Stopped) is reached, by default the stages before it should
 * be passed not including pre-resume stages.
 * for example:
 * Stopped => 00100000 (binary)
 * When reached, we need to cumulate the previous stages in the final representation.
 * Hence the output would be 00110000 (binary) including Paused stage & not including any pre-resume stage
 * example 2:
 * Stage => Detached = 01000000 (binary)
 * output => 01110000 (cumulative binary)
 */
@VisibleForTesting
fun Int.toPostPauseCumulative(): Int = this or trailingMask(this.countTrailingZeroBits(), 0x70)

/**
 * Given a number of bits that needs to 1's for masking, this method takes the number of bits
 * and returns an integer representing the binary 1's.
 * for example:
 * bitsCount = 5
 * output = 31 binary(11111)
 * example 2:
 * given a finale
 * bitsCount = 5
 * finale = 28 binary (11100)
 * output = binary (11100) instead of (11111)
 */
private fun trailingMask(bitsCount: Int, finale: Int = 0xFF): Int {
    var mask = 0x00
    repeat(bitsCount) { index -> mask = mask or (0x01 shl index) }
    return mask and finale
}

interface ComposeLifeCycleMonitor {
    fun onEnteringComposition(
        activityId: Int,
        id: Int,
        screenName: String,
        view: View?
    ): ComposeNode?

    fun onOwnerStarted(node: ComposeNode)
    fun onOwnerResumed(node: ComposeNode)
    fun onOwnerPaused(node: ComposeNode)
    fun onOwnerStopped(node: ComposeNode)
    fun onExitingComposition(node: ComposeNode)
}

private const val addingComposeNodeErrorMessage =
    "Something went wrong while adding started compose screen"

private const val resumingComposeNodeErrorMessage =
    "Something went wrong while adding resuming compose screen"

private const val pausingComposeNodeErrorMessage =
    "Something went wrong while adding pausing compose screen"

private const val removingComposeNodeErrorMessage =
    "Something went wrong while removing disposed compose screen"

class IBGComposeLifeCycleMonitor(
    private val root: ScreenChildrenOwner,
    private val reproStepsCaptor: ReproStepsCaptor
) : ComposeLifeCycleMonitor {

    override fun onEnteringComposition(
        activityId: Int,
        id: Int,
        screenName: String,
        view: View?
    ): ComposeNode? = runOrLogError(errorMessage = addingComposeNodeErrorMessage) {
        findParent(activityId, view)?.run {
            ComposeNode.Factory.create(id, screenName, this)
                .also(this::addChild)
        }
    }.getOrNull()

    override fun onOwnerStarted(node: ComposeNode) {
        runOrLogError(errorMessage = addingComposeNodeErrorMessage) {
            addStep(StepType.COMPOSE_STARTED, node.simpleName)
        }
    }

    override fun onOwnerResumed(node: ComposeNode) {
        runOrLogError(errorMessage = resumingComposeNodeErrorMessage) {
            node.also(ScreenNode::activate)
                .apply { addParentResumeStep(parent) }
                .apply { addStep(StepType.COMPOSE_RESUMED, simpleName) }
        }
    }

    override fun onOwnerPaused(node: ComposeNode) {
        runOrLogError(errorMessage = pausingComposeNodeErrorMessage) {
            node.also(ScreenNode::deactivate)
                .apply { addStep(StepType.COMPOSE_PAUSED, simpleName) }
        }
    }

    override fun onOwnerStopped(node: ComposeNode) {
        runOrLogError(errorMessage = addingComposeNodeErrorMessage) {
            addStep(StepType.COMPOSE_STOPPED, node.simpleName)
        }
    }

    override fun onExitingComposition(node: ComposeNode) {
        runOrLogError(errorMessage = removingComposeNodeErrorMessage) {
            node.apply { parent?.removeChild(id) }
                .apply(ComposeNode::clean)
                .apply { addStep(StepType.COMPOSE_DISPOSED, simpleName) }
        }
    }

    private fun addStep(@StepType stepType: String, screenName: String) {
        if (!isInstabugInEnabledState()) return
        reproStepsCaptor
            .addVisualUserStep(stepType, screenName, screenName, null)
    }

    private fun addParentResumeStep(parent: ScreenChildrenOwner?) {
        parent?.let { it as? FragmentNode }
            ?.let { FragmentsReproStepsHandler.addParentReproStep(Resumed, it) }
    }

    private fun findParent(activityId: Int, view: View?): ScreenChildrenOwner? {
        val activityParent = (root.findChild(activityId) as? ScreenChildrenOwner) ?: return null
        val viewParentsIds = extractViewParentIds(view)
        val potentialParents = Array<ScreenChildrenOwner?>(5) { null }
        ScreenChildrenOwner.Helper.traverseBreadth(
            activityParent,
            criteria = { it is FragmentNode }
        ) { examination ->
            (examination as? FragmentNode)?.let { fragmentNode ->
                fragmentNode.viewId
                    .let { vid -> viewParentsIds.indexOf(vid) }
                    .takeUnless { matchingIndex -> matchingIndex == -1 }
                    ?.also { matchingIndex -> potentialParents[matchingIndex] = fragmentNode }
            }
        }
        return potentialParents.filterNotNull().firstOrNull() ?: activityParent
    }

    private fun extractViewParentIds(view: View?): IntArray {
        val parents = IntArray(5) { -2 }
        var currentParent: ViewParent = view?.parent ?: return parents
        parents[0] = currentParent.hashCode()
        for (index in 1 until 5) {
            currentParent = currentParent.parent ?: break
            parents[index] = currentParent.hashCode()
        }
        return parents
    }
}

@VisibleForTesting
fun isInstabugInEnabledState(): Boolean =
    InstabugStateProvider.getInstance().state == InstabugState.ENABLED
