package com.instabug.library.networkinterception.util

private const val LC_CONTENT_TYPE_KEY = "content-type"
private const val LC_CONTENT_LENGTH_KEY = "content-length"
private const val LC_MULTI_PART_PREFIX = "multipart"
private const val LC_PROTO_BUF_CONTENT_TYPE = "application/protobuf"
private const val LC_PROTO_BUF_X_CONTENT_TYPE = "application/x-protobuf"
private const val LC_OCTET_STREAM_CONTENT_TYPE = "application/octet-stream"
private const val LC_CONTENT_ENCODING_HEADER_KEY = "content-encoding"
private const val LC_GZIPPED_CONTENT_ENCODING = "gzip"
private const val LC_APOLLO_GQL_QUERY_HEADER_KEY = "x-apollo-operation-name"
private const val LC_IDENTITY_CONTENT_ENCODING = "identity"
private const val LC_ACCEPT_RANGES_HEADER_KEY = "accept-ranges"
private const val LC_JSON_CONTENT_TYPE = "application/json"

/**
 * creates a new map from the passed map but with it's keys in lower case
 */
fun Map<String, String>.toLowerCaseKeys() = mapKeys { it.key.lowercase() }

/**
 * For headers map with lowercase keys, it returns the content-type
 */
fun Map<String, String>.lcContentType(): String? = get(LC_CONTENT_TYPE_KEY)

/**
 * For headers map with lowercase keys, it returns the content-length
 */
fun Map<String, String>.lcContentLength(): Long? = get(LC_CONTENT_LENGTH_KEY)?.toLongOrNull()

/**
 * For headers map with lowercase keys, it returns whether the request is multipart or not
 */
fun Map<String, String>.lcIsMultiPart(): Boolean =
    lcContentType()?.startsWith(LC_MULTI_PART_PREFIX, true) ?: false

/**
 * For headers map with lowercase keys, it returns whether the request is protocol buffer or not
 */
fun Map<String, String>.lcIsProtobuf(): Boolean =
    lcContentType()?.run {
        contains(LC_PROTO_BUF_CONTENT_TYPE, true) ||
                contains(LC_PROTO_BUF_X_CONTENT_TYPE, true)
    } ?: false

/**
 * For headers map with lowercase keys, it returns whether the request is file download or not
 */
fun Map<String, String>.lcIsFileDownload(): Boolean =
    lcIsOctetStream() || (lcHasAcceptRangesHeader() && !lcIsJsonContentType())

/**
 * For headers map with lowercase keys, it returns whether the request content-type is octet-stream or not
 */
fun Map<String, String>.lcIsOctetStream(): Boolean =
    lcContentType()?.contains(LC_OCTET_STREAM_CONTENT_TYPE, true) ?: false

/**
 * For headers map with lowercase keys, it returns the content-encoding
 */
fun Map<String, String>.lcContentEncoding() = get(LC_CONTENT_ENCODING_HEADER_KEY)

/**
 * For headers map with lowercase keys, it returns whether the body is encoded or not
 */
fun Map<String, String>.lcIsContentEncoded() =
    lcContentEncoding()?.equals(LC_IDENTITY_CONTENT_ENCODING, true) == false

/**
 * For headers map with lowercase keys, it returns whether the body is gzipped or not
 */
fun Map<String, String>.lcIsGZipped() =
    lcContentEncoding()?.equals(LC_GZIPPED_CONTENT_ENCODING, true) ?: false

/**
 * For headers map with lowercase keys, it returns Apollo GraphQL query name
 */
fun Map<String, String>.lcApolloGQLQuery() = get(LC_APOLLO_GQL_QUERY_HEADER_KEY)

/**
 * For headers map with lowercase keys, it returns whether the body is accept-ranges or not
 */
fun Map<String, String>.lcHasAcceptRangesHeader() =
    this.containsKey(LC_ACCEPT_RANGES_HEADER_KEY)

/**
 * For headers map with lowercase keys, it returns whether the body is json or not
 */
fun Map<String, String>.lcIsJsonContentType() =
    lcContentType()?.contains(LC_JSON_CONTENT_TYPE, true) ?: false
