package com.instabug.library.networkinterception.util

import com.instabug.library.networkinterception.delegate.Utf8CodePointEvaluator
import com.instabug.library.util.getByte

/**
 * Validate if a UTF-8 code Int represents a plain text
 */
fun Int.isUtf8PlainText() =
    !Character.isISOControl(this) || Character.isWhitespace(this)

/**
 * Decode UTF-8 byte array to utf-8 code point
 * and validate if the result is plain text or not
 */
fun Utf8CodePointEvaluator.isPlainText(
    bytes: ByteArray,
    off: Int = 0,
    len: Int = bytes.size
): Boolean {
    for (index in off until (off + len)) {
        addAndValidateByte(bytes[index].toUByte())
            .takeUnless { it }
            ?.let { return false }
    }
    return true
}

/**
 * Decode UTF-8 bytes encoded in an int to utf-8 code point
 * and validate if the result is plain text or not
 */
fun Utf8CodePointEvaluator.isPlainText(bytes: Int): Boolean {
    for (byteIndex in 0 until Int.SIZE_BYTES) {
        bytes.getByte(byteIndex)?.toUByte()?.let { byte ->
            addAndValidateByte(byte)
                .takeUnless { it }
                ?.let { return false }
        }
    }
    return true
}

/**
 * Decode UTF-8 bytes encoded in an int to utf-8 code point
 * and validate if the result is plain text or not
 */
fun Utf8CodePointEvaluator.isPlainText(byte: Byte): Boolean =
    addAndValidateByte(byte.toUByte())

/**
 * Add the byte to the code point evaluator and validate the result if it was plain text or not
 */
private fun Utf8CodePointEvaluator.addAndValidateByte(byte: UByte): Boolean {
    add(byte)
    return getCodePoint()?.isUtf8PlainText() != false
}
