package com.instabug.library.networkinterception.urlconnection

import com.instabug.library.networkinterception.NetworkInterceptionServiceLocator
import com.instabug.library.transform.TransformationClass
import com.instabug.library.util.extenstions.runOrLogError
import java.net.HttpURLConnection
import java.net.URL
import java.net.URLConnection
import javax.net.ssl.HttpsURLConnection

@TransformationClass
internal object IBGUrlConnectionHandler {

    @JvmStatic
    fun openMonitoredConnection(url: URL): URLConnection =
        url.openConnection().run {
            runOrLogError {
                takeIf { url.isBlackListed() } ?: intercept(url)
            }.getOrNull() ?: this
        }

    private fun URLConnection.intercept(url: URL) =
        when (this) {
            is HttpsURLConnection ->
                NetworkInterceptionServiceLocator.ibgHttpsUrlConnectionFactory
                    .create(Pair(url, this))

            is HttpURLConnection ->
                NetworkInterceptionServiceLocator.ibgHttpUrlConnectionFactory
                    .create(Pair(url, this))

            else -> null
        }

    private fun URL.isBlackListed() =
        isInstabugURL() &&
                (isIBGFeaturesRequest() || isIBGNonFatalsRequest() || isIBGCrashesRequest())

    private fun URL.isIBGFeaturesRequest() = path.contains("api/sdk/v3/features")
    private fun URL.isIBGNonFatalsRequest() = path.contains("api/sdk/v3/diagnostics")
    private fun URL.isIBGCrashesRequest() = path.endsWith("api/sdk/v3/crashes")
    private fun URL.isInstabugURL() = this.host.contains("instabug", true)
}

