package com.instabug.library.networkinterception.sanitization

import com.instabug.library.Constants
import com.instabug.library.internal.servicelocator.Provider
import com.instabug.library.logging.listeners.networklogs.NetworkLogListener
import com.instabug.library.logging.listeners.networklogs.NetworkLogSnapshot
import com.instabug.library.fill.Filler
import com.instabug.library.map.Mapper
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.runOrLogError

class UserDefinedSanitizer(
    private val networkLogListenerProvider: Provider<NetworkLogListener?>,
    private val networkLogBuilderToSnapshotMapper: Mapper<NetworkLogModel.Builder, NetworkLogSnapshot>,
    private val networkSnapshotToBuilderFiller: Filler<NetworkLogModel.Builder, NetworkLogSnapshot>,
    private val configurationProvider: IBGNetworkInterceptionConfigurationProvider
) : Sanitizer<NetworkLogModel.Builder> {

    companion object {
        const val REMOVING_NETWORK_LOG_URL_IS_NOT_ALLOWED: String =
            "Removing URL property from the network request is not allowed." +
                 " Please contact support for more information."
        const val MANUAL_SANITIZATION_DISABLED: String =
            "Manual network sanitization is not applied, feature is disabled"
    }

    override fun sanitize(item: NetworkLogModel.Builder): NetworkLogModel.Builder? = runOrLogError {
        networkLogListenerProvider()
            ?.takeIf { validateAndLogFeatureState() }
            .let { nullableListener ->
                when (nullableListener) {
                    null -> item
                    else -> nullableListener.sanitize(item)
                }
            }
    }.getOrNull()

    private fun validateAndLogFeatureState(): Boolean =
        configurationProvider.v2ManualSanitizationEnabled.apply {
            if (!this) {
                InstabugSDKLogger.e(Constants.LOG_TAG, MANUAL_SANITIZATION_DISABLED)
            }
        }

    private fun NetworkLogListener.sanitize(
        item: NetworkLogModel.Builder
    ): NetworkLogModel.Builder? =
        onNetworkLogCapturedCatching(networkLogBuilderToSnapshotMapper.map(item))
            ?.takeIf { it.isValid() }
            ?.let { modifiedSnapshot ->
                item.apply {
                    with(networkSnapshotToBuilderFiller) {
                        fill(modifiedSnapshot)
                    }
                }
            }

    private fun NetworkLogListener.onNetworkLogCapturedCatching(
        snapshot: NetworkLogSnapshot
    ) = runCatching { onNetworkLogCaptured(snapshot) }
        .onFailure {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Error while invoking NetworkLogListener",
                it
            )
        }
        .getOrNull()

    fun NetworkLogSnapshot.isValid(): Boolean = (!url.isNullOrBlank()).also {
        if (!it) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                REMOVING_NETWORK_LOG_URL_IS_NOT_ALLOWED
            )
        }
    }
}
