package com.instabug.library.networkinterception.okhttp.model

import androidx.annotation.IntDef
import com.instabug.library.model.TimeCapture
import com.instabug.library.networkinterception.model.NetworkClient
import com.instabug.library.networkinterception.model.NetworkLogModel

class OkHttpNetworkLogModel : NetworkLogModel {

    private constructor(@NetworkClient clientId: Int, url: String) : super(clientId, url)

    private constructor(origin: NetworkLogModel) : super(origin) {
        if (origin is OkHttpNetworkLogModel) {
            events = origin.events
        }
    }

    var events: List<OkHttpEvent>? = null
        private set

    override fun newBuilder(): Builder = Builder(OkHttpNetworkLogModel(this))

    class Builder internal constructor(
        private var model: OkHttpNetworkLogModel
    ) : NetworkLogModel.Builder(model) {

        constructor(
            @NetworkClient clientId: Int,
            url: String
        ) : this(OkHttpNetworkLogModel(clientId, url))

        constructor(
            networkLogModel: NetworkLogModel
        ) : this(OkHttpNetworkLogModel(networkLogModel))

        fun setOkHttpEvents(events: List<OkHttpEvent>?): Builder = apply {
            model.events = events
        }
        val okHttpEvents
            get(): List<OkHttpEvent>? = model.events

        /**
         * Only accept new timestamp as long as it's less than the current timestamp
         */
        @Synchronized
        override fun setStartTimeStampMs(timeStamp: Long) = apply {
            timeStamp.takeIf { startTimestampMs <= 0 || it < startTimestampMs }
                ?.let { super.setStartTimeStampMs(it) }
        }

        /**
         * Only accept new duration as long as it's larger than the current duration
         */
        override fun setDurationMus(duration: Long) = apply {
            duration.takeIf { duration > durationMus }
                ?.let { super.setDurationMus(duration) }
        }

        override fun build(): OkHttpNetworkLogModel = model.apply {
            request = requestBuilder?.build()
            response = responseBuilder?.build()
        }
    }
}

data class OkHttpEvent(@OkHttpEventId val id: Int, val timeCapture: TimeCapture)

@Retention(AnnotationRetention.SOURCE)
@IntDef(
    OkHttpEventId.DNS_START,
    OkHttpEventId.DNS_END,
    OkHttpEventId.CONNECT_START,
    OkHttpEventId.SECURE_CONNECT_START,
    OkHttpEventId.SECURE_CONNECT_END,
    OkHttpEventId.CONNECT_END,
    OkHttpEventId.CONNECT_FAILED,
    OkHttpEventId.REQUEST_HEADERS_START,
    OkHttpEventId.REQUEST_HEADERS_END,
    OkHttpEventId.REQUEST_BODY_START,
    OkHttpEventId.REQUEST_BODY_END,
    OkHttpEventId.REQUEST_FAILED,
    OkHttpEventId.RESPONSE_HEADERS_START,
    OkHttpEventId.RESPONSE_HEADERS_END,
    OkHttpEventId.RESPONSE_BODY_START,
    OkHttpEventId.RESPONSE_BODY_END,
    OkHttpEventId.RESPONSE_FAILED,
    OkHttpEventId.CALL_END,
    OkHttpEventId.CALL_FAILED,
)
annotation class OkHttpEventId {
    companion object {
        const val DNS_START = 0
        const val DNS_END = 1
        const val CONNECT_START = 2
        const val SECURE_CONNECT_START = 3
        const val SECURE_CONNECT_END = 4
        const val CONNECT_END = 5
        const val CONNECT_FAILED = 6
        const val REQUEST_HEADERS_START = 7
        const val REQUEST_HEADERS_END = 8
        const val REQUEST_BODY_START = 9
        const val REQUEST_BODY_END = 10
        const val REQUEST_FAILED = 11
        const val RESPONSE_HEADERS_START = 12
        const val RESPONSE_HEADERS_END = 13
        const val RESPONSE_BODY_START = 14
        const val RESPONSE_BODY_END = 15
        const val RESPONSE_FAILED = 16
        const val CALL_END = 17
        const val CALL_FAILED = 18
    }
}
