package com.instabug.library.networkinterception.model

import androidx.annotation.IntDef

@Retention(AnnotationRetention.SOURCE)
@IntDef(
    NetworkClient.OKHTTP,
    NetworkClient.OkHTTP_GQL,
    NetworkClient.URL_CONNECTION,
    NetworkClient.GRPC,
    NetworkClient.CP,
    NetworkClient.CP_GQL,
    NetworkClient.CP_GRPC,
    NetworkClient.MANUAL
)
annotation class NetworkClient {
    companion object {
        const val OKHTTP = 0
        const val OkHTTP_GQL = 1
        const val URL_CONNECTION = 2
        const val GRPC = 3
        const val CP = 4
        const val CP_GQL = 5
        const val CP_GRPC = 6
        const val MANUAL = 7
    }
}

open class NetworkLogModel protected constructor(
    @NetworkClient var clientId: Int,
    pUrl: String
) {

    protected constructor(origin: NetworkLogModel) : this(origin.clientId, origin.url) {
        startTimestampMs = origin.startTimestampMs
        durationMus = origin.durationMus
        injectableHeaders = origin.injectableHeaders
        request = origin.request
        response = origin.response
        userModified = origin.userModified
    }

    var url: String = pUrl
        protected set

    var startTimestampMs: Long = -1
        protected set

    var durationMus: Long = 0
        protected set

    var injectableHeaders: NetworkLogW3CExternalTraceIdInfo? = null
        protected set

    var request: NetworkLogRequestModel? = null
        protected set

    var response: NetworkLogResponseModel? = null
        protected set

    var userModified: Boolean = false
        protected set

    open fun newBuilder(): Builder =
        Builder(NetworkLogModel(this))

    open class Builder internal constructor(
        private var holder: NetworkLogModel
    ) {
        constructor(
            @NetworkClient clientId: Int,
            url: String
        ) : this(NetworkLogModel(clientId, url))

        @NetworkClient
        val clientId = holder.clientId
        fun setClient(@NetworkClient client: Int) = apply {
            holder.clientId = client
        }

        val url: String
            get() = holder.url
        fun setUrl(url: String): Builder = apply {
            holder.url = url
        }

        open fun setStartTimeStampMs(timeStamp: Long): Builder = apply {
            holder.startTimestampMs = timeStamp
        }

        val startTimestampMs: Long
            get() = holder.startTimestampMs

        open fun setDurationMus(duration: Long): Builder = apply {
            holder.durationMus = duration
        }

        val durationMus: Long
            get() = holder.durationMus

        fun setInjectableHeaders(injectableHeaders: NetworkLogW3CExternalTraceIdInfo?): Builder =
            apply {
                holder.injectableHeaders = injectableHeaders
            }

        val injectableHeaders: NetworkLogW3CExternalTraceIdInfo?
            get() = holder.injectableHeaders

        var requestBuilder: NetworkLogRequestModel.Builder? = holder.request?.newBuilder()
            protected set

        fun setRequestBuilder(requestBuilder: NetworkLogRequestModel.Builder?): Builder = apply {
            this.requestBuilder = requestBuilder
        }

        var responseBuilder: NetworkLogResponseModel.Builder? = holder.response?.newBuilder()
            private set

        fun setResponseBuilder(responseBuilder: NetworkLogResponseModel.Builder?): Builder = apply {
            this.responseBuilder = responseBuilder
        }

        fun setUserModified(userModified: Boolean): Builder = apply {
            holder.userModified = userModified
        }

        val userModified: Boolean
            get() = holder.userModified

        open fun build(): NetworkLogModel = holder.apply {
            request = requestBuilder?.build()
            response = responseBuilder?.build()
        }
    }
}
