package com.instabug.library.networkinterception.externaltrace

import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.externaltrace.W3CExternalNetworkTraceIdFactory.Companion.TraceParentHeaderKey
import com.instabug.library.networkinterception.model.NetworkLogModel
import com.instabug.library.networkinterception.model.NetworkLogW3CExternalTraceIdInfo
import com.instabug.library.sanitize.Sanitizer
import com.instabug.library.util.extenstions.runOrLogError

class W3CExternalHeaderSanitizer(
    private val configurations: IBGNetworkInterceptionConfigurationProvider,
    private val injectableHeaderFactory: ParameterizedFactory<NetworkLogW3CExternalTraceIdInfo?, Map<String, String>?>
) : Sanitizer<NetworkLogModel.Builder> {

    override fun sanitize(item: NetworkLogModel.Builder): NetworkLogModel.Builder = item.apply {
        runOrLogError {
            if (configurations.isW3CNetworkExternalTraceIdEnabled()) {
                applySanitization()
            }
        }
    }

    private fun NetworkLogModel.Builder.applySanitization() = apply {
        injectableHeaders?.takeIf { responseBuilder?.clientSideThrowable == null }
            ?.also { regenerateInjectableHeadersInfoIfPossible() }
            ?: setInjectableHeaders(null)
    }

    private fun NetworkLogModel.Builder.regenerateInjectableHeadersInfoIfPossible() {
        if (shouldSanitizeInjectableHeaders()) regenerateInjectableHeadersInfo()
    }

    private fun NetworkLogModel.Builder.shouldSanitizeInjectableHeaders() =
        injectableHeaders?.fullyGeneratedId?.let { generateId ->
            getTraceParentHeader()?.let { injectedHeader ->
                generateId != injectedHeader
            }
        } ?: false

    private fun NetworkLogModel.Builder.getTraceParentHeader() =
        requestBuilder?.headers?.get(TraceParentHeaderKey)

    private fun NetworkLogModel.Builder.regenerateInjectableHeadersInfo() {
        setInjectableHeaders(injectableHeaderFactory.create(requestBuilder?.headers))
    }
}
