package com.instabug.library.networkinterception.delegate.extract

import com.instabug.library.networkinterception.delegate.Utf8CodePointEvaluator
import com.instabug.library.networkinterception.util.isPlainText
import com.instabug.library.util.extenstions.logDebug
import com.instabug.library.util.extenstions.runOrLogError
import java.io.ByteArrayOutputStream

class PlainTextByteArrayOutputStream(
    val sampleSizeBytes: Int,
    val maxAllowedSizeBytes: Long,
    private val utf8CodePointEvaluator: Utf8CodePointEvaluator
) : ByteArrayOutputStream() {

    private var markedByteCount: Long? = null

    var isPlainText: Boolean = true
        private set

    var exceededMaxSize: Boolean = false
        private set

    var byteCount: Long = 0
        private set

    override fun write(b: Int) {
        runOrLogError {
            b.takeIf { it.validate() }?.let { super.write(b) }
            byteCount += 1
        }
    }

    override fun write(b: ByteArray) = write(b, 0, b.size)

    override fun write(b: ByteArray, off: Int, len: Int) {
        runOrLogError {
            b.takeIf { it.validateRange(off, len) }
                ?.let { byteArray ->
                    byteArray.takeIf { it.shouldRead(off, len) }?.let { super.write(b, off, len) }
                    byteCount += len
                }
        }
    }

    fun mark() {
        markedByteCount = byteCount
    }

    fun unmark() {
        markedByteCount = null
    }

    fun resetToMark() {
        runOrLogError {
            markedByteCount?.also { currentMark ->
                byteCount = currentMark
                count = currentMark.toInt().takeIf { it <= count } ?: count
                markedByteCount = null
            } ?: let {
                reset()
                byteCount = 0
            }
        }
    }
    private inline fun validate(newContentSize: Int, plainTextValidator: () -> Boolean): Boolean =
        when {
            exceededMaxSize -> {
                false
            }

            !isPlainText -> {
                false
            }

            willExceedMaxSize(newContentSize) -> {
                exceededMaxSize = true
                reset()
                false
            }

            byteCount > sampleSizeBytes -> {
                isPlainText
            }

            else -> {
                plainTextValidator.invoke().also {
                    isPlainText = it
                    if (!it) {
                        reset()
                        "Body is not plain text, skipping body collection".logDebug()
                    }
                }
            }
        }

    private fun willExceedMaxSize(newContentSize: Int) = newContentSize + byteCount > maxAllowedSizeBytes

    private fun ByteArray.shouldRead(off: Int, len: Int): Boolean =
        validate(len) {
            (sampleSizeBytes - byteCount)
                .takeIf { it > 0 }
                ?.let { remainingSampleSize ->
                    utf8CodePointEvaluator.isPlainText(this, off,
                        kotlin.math.min(remainingSampleSize.toInt(), len)
                    )
                } ?: true
        }

    private fun ByteArray.validateRange(off: Int, len: Int): Boolean =
        off in 0 until size && len > 0 && (off + len) <= size

    private fun Int.validate(): Boolean =
        validate(1) { utf8CodePointEvaluator.isPlainText(this.toByte()) }

}
