package com.instabug.library.networkinterception.delegate.extract

import com.instabug.library.networkinterception.config.IBGNetworkInterceptionConfigurationProvider
import com.instabug.library.networkinterception.delegate.validate.generateLargeBodyMessage
import com.instabug.library.sanitize.Validator
import com.instabug.library.util.extenstions.logDebug
import com.instabug.library.util.extenstions.runOrLogAndReport
import java.io.InputStream
import kotlin.math.min

class ImmediateInputStreamExtractor(
    private val sampleSize: Int,
    private val configurations: IBGNetworkInterceptionConfigurationProvider,
    private val plainTextValidator: Validator<ByteArray>
) : NetworkBodyExtractor<InputStream> {

    private val maxAllowedBytes: Long
        get() = configurations.v2MaxAllowedBodySizeBytes

    override fun extract(
        body: InputStream,
        onBodyReady: (body: String?, length: Long?) -> Unit
    ): InputStream? = body.use {
        runCatching {
            it.takeIf { it.available() <= maxAllowedBytes }?.also { autoClosableBody ->
                val sample = autoClosableBody.readSample()
                val available = autoClosableBody.available()
                val totalLength = sample.size + available
                if (plainTextValidator.isValid(sample)) {
                    val allBytes = sample.expand(totalLength)
                    autoClosableBody.readRemainingRecursively(allBytes, sample.size)
                    val plainTextBody = String(allBytes, Charsets.UTF_8)
                    onBodyReady.invoke(plainTextBody, totalLength.toLong())
                } else {
                    "Body is not plain text, skipping body collection".logDebug()
                    onBodyReady.invoke(null, totalLength.toLong())
                }
            } ?: run {
                onBodyReady.invoke(generateLargeBodyMessage(maxAllowedBytes), it.available().toLong())
            }
        }.runOrLogAndReport(
            "ImmediateInputStreamExtractor: Failed to extract Body input stream",
            true
        )
        null
    }

    private fun InputStream.readSample(): ByteArray {
        val minSampleSize = min(sampleSize, available())
        val sample = ByteArray(minSampleSize)
        read(sample, 0, minSampleSize)
        return sample
    }

    private fun InputStream.readRemainingRecursively(
        buffer: ByteArray,
        offset: Int,
        iterationLimit: Int = 5
    ) {
        val available = available()
        if (available <= 0) {
            return
        }
        if (offset >= buffer.size) {
            return
        }
        if (buffer.size < offset + available) {
            return
        }
        if (iterationLimit <= 0) {
            return
        }
        val readCount = read(buffer, offset, available)
        readRemainingRecursively(buffer, offset + readCount, iterationLimit - 1)
    }

    private fun ByteArray.expand(newSize: Int) = this.let { original ->
        ByteArray(newSize) { index ->
            if (index < original.size) {
                original[index]
            } else {
                0
            }
        }
    }
}
