package com.instabug.library.networkinterception.config

import com.instabug.library.featuresflags.configs.FeatureFlagsConfigsHandler
import com.instabug.library.internal.crossplatform.ICPConfigurationsProvider
import com.instabug.library.util.addFlag
import org.json.JSONArray
import org.json.JSONObject

class NetworkInterceptionConfigurationHandler(
    private val configurationProvider: IBGNetworkInterceptionConfigurationProvider,
    private val cpConfigurationsProvider: ICPConfigurationsProvider
) : FeatureFlagsConfigsHandler {

    override fun handleConfigs(featuresResponse: JSONObject) {
        featuresResponse.optJSONObject(KEY_BE_NETWORK_CONFIG)?.also { networkObject ->
            handleV2InterceptionConfigurations(networkObject)
            handleW3cConfigurations(networkObject)
            handleNetworkBodyCaptureRules(networkObject)
            networkObject.optJSONObject(KEY_BE_AUTO_MASKING).also { autoMaskingObject ->
                handleAutoMaskingConfigurations(autoMaskingObject)
            }
            cpConfigurationsProvider.onFeaturesStateChanged()
        } ?: reset()
        featuresResponse.optLong(
            KEY_BE_MAX_BODY_SIZE,
            DEFAULT_V2_MAX_BODY_SIZE
        ).let { configurationProvider.v2MaxAllowedBodySizeBytes = it }
    }

    override fun reset() {
        configurationProvider.reset()
        cpConfigurationsProvider.onFeaturesStateChanged()
    }

    private fun handleW3cConfigurations(networkObject: JSONObject) {
        networkObject.optDouble(
            KEY_BE_W3C_EXTERNAL_TRACE_ID_AVAILABLE,
            DEFAULT_W3C_EXTERNAL_TRACE_ID_PERCENTAGE
        ).let(configurationProvider::setW3CNetworkExternalTraceIdFeatureAvailabilityPercentage)

        networkObject.optBoolean(
            KEY_BE_ATTACHING_GENERATE_W3C_TRACE_ID_AVAILABLE,
            DEFAULT_GENERATED_W3C_ATTACHING_AVAILABLE
        ).let { configurationProvider.isAttachingGeneratedW3CExternalTraceIdFeatureAvailable = it }

        networkObject.optBoolean(
            KEY_BE_ATTACHING_CAPTURED_W3C_TRACE_ID_AVAILABLE,
            DEFAULT_CAPTURED_W3C_ATTACHING_AVAILABLE
        ).let { configurationProvider.isAttachingCapturedW3CExternalTraceIdFeatureAvailable = it }
    }

    private fun handleV2InterceptionConfigurations(networkObject: JSONObject) {
        networkObject.optJSONObject(KEY_BE_V2_INTERCEPTION_CONFIG)?.let { v2Config ->
            v2Config.optDouble(
                KEY_BE_ENABLED, DEFAULT_V2_INTERCEPTION_PERCENTAGE
            ).let(configurationProvider::setV2InterceptionFeatureAvailabilityPercentage)
            v2Config.optV2InterceptorFlag(KEY_BE_OK_HTTP_CONFIG).let {
                configurationProvider.v2OkHttpInterceptorBEState = it
            }
            v2Config.optV2InterceptorFlag(KEY_BE_URL_CONNECTION_CONFIG).let {
                configurationProvider.v2UrlConnectionInterceptorBEState = it
            }
            v2Config.optV2InterceptorFlag(KEY_BE_GRPC_CONFIG).let {
                configurationProvider.v2GrpcInterceptorBEState = it
            }
            v2Config.optBoolean(KEY_BE_GQL_ENABLED, DEFAULT_V2_GQL_FEATURE_AVAILABLE)
                .let { configurationProvider.v2GqlFeatureAvailable = it }
            handleV2SanitizationConfigurations(v2Config)
        } ?: configurationProvider.resetV2InterceptionFlags()
    }

    private fun JSONObject.optV2InterceptorFlag(interceptorKey: String): Int {
        return optJSONObject(interceptorKey)?.let {
            var flag = 0
            if (it.optBoolean(KEY_BE_ENABLED, DEFAULT_V2_INTERCEPTOR_AVAILABLE)) {
                flag = flag.addFlag(V2InterceptorFlags.ENABLED)
            }
            if (it.optBoolean(KEY_BE_REQUEST_BODY_COLLECTION, DEFAULT_V2_REQUEST_BODY_INTERCEPTION_AVAILABLE)) {
                flag = flag.addFlag(V2InterceptorFlags.REQUEST_BODY_COLLECTION_ENABLED)
            }
            if (it.optBoolean(KEY_BE_RESPONSE_BODY_COLLECTION, DEFAULT_V2_RESPONSE_BODY_INTERCEPTION_AVAILABLE)) {
                flag = flag.addFlag(V2InterceptorFlags.RESPONSE_BODY_COLLECTION_ENABLED)
            }
            flag
        } ?: DEFAULT_V2_INTERCEPTOR_STATE
    }

    private fun handleV2SanitizationConfigurations(v2Config: JSONObject) {
        v2Config.optJSONObject(KEY_BE_SANITIZATION_CONFIG)?.let { sanitizationObject ->
            sanitizationObject.optBoolean(
                KEY_BE_MANUAL_SANITIZATION_ENABLED,
                DEFAULT_V2_MANUAL_SANITIZATION_AVAILABLE
            ).let { configurationProvider.v2ManualSanitizationAvailable = it }
            sanitizationObject.optStringSet(
                KEY_BE_INSTABUG_SANITIZATION_QUERY_PARAMS,
                DEFAULT_V2_INSTABUG_SANITIZATION_QUERY_PARAMS
            ).let { configurationProvider.v2InstabugSanitizationQueryParams = it }
        } ?: configurationProvider.resetV2SanitizationFlags()
    }

    private fun JSONObject.optStringSet(key: String, defaultValue: Set<String>): Set<String> =
        optJSONArray(key)?.let { jsonStringArray ->
            HashSet<String>().also { set ->
                for (index in 0 until jsonStringArray.length()) {
                    set += jsonStringArray.getString(index)
                }
            }
        } ?: defaultValue

    private fun handleAutoMaskingConfigurations(autoMaskingJsonObject: JSONObject?) {
        autoMaskingJsonObject?.let { autoMaskingObject ->
            autoMaskingObject
                .optDouble(
                    KEY_BE_AUTO_MASKING_AVAILABLE,
                    DEFAULT_AUTO_MASKING_PERCENTAGE,
                ).let(configurationProvider::setAutoMaskingFeatureAvailabilityPercentage)

            autoMaskingObject
                .optJSONArray(
                    KEY_BE_AUTO_MASKING_HEADER_KEYS,
                ).let(this::parseBEAutoMaskingHeaderKeys)

            autoMaskingObject
                .optJSONArray(
                    KEY_BE_AUTO_MASKING_QUERY_KEYS,
                ).let(this::parseBEAutoMaskingQueryKeys)
        } ?: run {
            configurationProvider.resetAutoMasking()
        }
    }
    private fun handleNetworkBodyCaptureRules(networkObject: JSONObject) =
        networkObject.optBoolean(KEY_BE_CAPTURE_NETWORK_BODY_LOGS, DEFAULT_CAPTURE_NETWORK_BODY_LOG_BE)
            .let {
                configurationProvider.isNetworkLogBodyEnabledByBE = it
            }

    private fun parseBEAutoMaskingHeaderKeys(headerKeys: JSONArray?) {
        configurationProvider.autoMaskingBEHeaderKeys = headerKeys?.toStringSet() ?: emptySet()
    }

    private fun parseBEAutoMaskingQueryKeys(queryKeys: JSONArray?) {
        configurationProvider.autoMaskingBEQueryKeys = queryKeys?.toStringSet() ?: emptySet()
    }

    private fun JSONArray.toStringSet() = HashSet<String>().also { set ->
        for (index in 0 until length()) {
            set.add(this[index]?.toString() ?: "")
        }
    }
}
