package com.instabug.library.logging.disklogs;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.internal.orchestrator.Action;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.storage.operation.DiskOperation;
import com.instabug.library.internal.storage.operation.DiskOperationCallback;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.memory.MemoryUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * A class to provide API for writing logs Async or sync
 */
public class WriteLogDiskOperator implements DiskOperation<Uri, Context> {


    private final File file;
    private String logData;
    private final boolean encrypted;

    /**
     * Construct the operator by giving it the logging file which will be used to write in
     * and list of logs to write them.
     *
     * @param file      log file
     * @param logData   logs
     * @param encrypted encrypt or not
     */
    public WriteLogDiskOperator(@NonNull File file, @NonNull String logData, boolean encrypted) {
        this.file = file;
        this.logData = logData;
        this.encrypted = encrypted;
    }

    /**
     * A method to execute writing operation sync
     *
     * @param context this context isn't used so it can be nullable.
     * @return a uri from file
     * @throws IOException
     */
    @Override
    public Uri execute(@Nullable Context context) throws IOException {
        try {
            writeFile(logData, context);
        } catch (IOException ex) {
            Log.e(Constants.LOG_TAG, "Error while writing logsto disk: ", ex);
        }
        return Uri.fromFile(file);
    }

    /**
     * A method to execute writing operation async
     *
     * @param context this context isn't used so it can be nullable.
     * @return a uri from file
     * @throws IOException
     */
    @Override
    public void executeAsync(final Context context, @Nullable final DiskOperationCallback<Uri> callback) {
        ActionsOrchestrator
                .obtainOrchestrator()
                .addWorkerThreadAction(new Action() {
                    @Override
                    public void run() throws Exception {
                        try {
                            writeFile(logData, context);
                        } catch (IOException ex) {
                            Log.e(Constants.LOG_TAG, "Error while writing logs to disk: ", ex);
                            if (callback != null) {
                                callback.onFailure(ex);
                            }
                        }
                        if (callback != null) {
                            callback.onSuccess(Uri.fromFile(file));
                        }
                    }
                })
                .orchestrate();
    }

    /**
     * Internal method to write logs in file
     * if memory is low then it will skip saving these logs
     *
     * @param logs    data
     * @param context
     * @throws IOException
     */

    private void writeFile(String logs, @Nullable Context context) throws IOException {
        if (context != null) {
            if (!MemoryUtils.isLowMemory(context)) {
                final FileOutputStream out = new FileOutputStream(file, true);
                try {
                    String encryptedLogs = encrypted ? EncryptionManager.encryptWithStaticKey(logs) : logs;
                    if (encryptedLogs != null) {
                        out.write(encryptedLogs.getBytes("UTF-8"));
                        out.write(EncryptionManager.LINE_FEED.getBytes("UTF-8"));
                    } else {
                        IBGDiagnostics.reportNonFatal(new Exception("Error writing logs exception"),
                                "Couldn't write logs to file due to error in encryption");
                    }
                } catch (Throwable throwable) {
                    InstabugSDKLogger.e(Constants.LOG_TAG, "Error while attamp to write log disk operator" + throwable.getMessage());
                    Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to " + throwable.getMessage());
                } finally {
                    out.close();
                }
            } else {
                Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to low memory");
            }
        } else {
            Log.e(Constants.LOG_TAG, "Couldn't write logs to disk due to null context");
        }
    }
}
