package com.instabug.library.logging.disklogs

import androidx.annotation.VisibleForTesting
import com.instabug.library.internal.resolver.LoggingSettingResolver
import com.instabug.library.internal.servicelocator.CoreServiceLocator
import com.instabug.library.model.LogData
import com.instabug.library.model.LogDescriptor
import com.instabug.library.model.LoggingSettings
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.runDefensive

class IBGLoggingThread(
    private val logsFileDiskOperator: FileDiskOperator = CoreServiceLocator.fileDiskOperator
) : Thread() {

    private val TAG = "IBGDiskLoggingThread"

    private val END_SESSION = "End-session"
    private var loggingInterval: Long =
        LoggingSettingResolver.getInstance().loggingSettings?.flushInterval
            ?: LoggingSettings.DEFAULT_FLUSH_INTERVAL_SECONDS * 1000L

    @Volatile
    @VisibleForTesting
    var logsStringBuilder = StringBuilder()

    @VisibleForTesting
    var wasInterrupted: Boolean = false

    init {
        start()
    }

    override fun run() {
        runDefensive {
            name = "IBGLoggingThread"
            while (LoggingSettingResolver.getInstance().loggingSettings?.level != LoggingSettings.LogLevels.NO_LOGS && !wasInterrupted) {
                try {
                    sleep(loggingInterval)
                } catch (e: InterruptedException) {
                    InstabugSDKLogger.v(TAG, "IBGDiskLoggingThread was interrupted")
                }

                if (logsStringBuilder.isNotEmpty()) {
                    writeLogs(LoggingSettingResolver.getInstance().loggingSettings?.isEncrypted == true)
                }
            }
        }
    }

    fun addLog(tag: String, msg: String, currentThread: String, timeStamp: Long) {
        val log = LogData.Builder()
            .setTag(tag)
            .setMessage(trimLogMessageIfNeeded(msg))
            .setCurrentThreadName(currentThread)
            .setTimestamp(timeStamp)
            .build()

        logsStringBuilder.append(log.toString())

        flushToDiskIfNeeded()
    }

    @VisibleForTesting
    fun flushToDiskIfNeeded() {
        if (hasExceededFlushCharLimit()) {
            val loggingSettings = LoggingSettingResolver.getInstance().loggingSettings
            writeLogs(loggingSettings?.isEncrypted == true)
        }
    }

    @VisibleForTesting
    fun hasExceededFlushCharLimit(): Boolean {
        return logsStringBuilder.length >= LoggingSettingResolver.getInstance().loggingSettings?.flushCharLimit ?: LoggingSettings.DEFAULT_FLUSH_CHAR_LIMIT
    }

    @VisibleForTesting
    fun trimLogMessageIfNeeded(msg: String): String {
        val limit = LoggingSettingResolver.getInstance().loggingSettings?.singleLogLimit
            ?: LoggingSettings.DEFAULT_SINGLE_LOG_LIMIT
        if (msg.length > limit) {
            val difference = msg.length - limit
            val msgBuilder = StringBuilder(msg)
            msgBuilder.delete(limit.toInt(), msg.length)
            msgBuilder.append("...$difference")
            return msgBuilder.toString()
        }
        return msg
    }

    @VisibleForTesting
    fun writeLogs(encrypted: Boolean) {
        logsFileDiskOperator.flush(encrypted, logsStringBuilder)
    }

    fun logSessionDescriptor(sessionDescriptor: LogDescriptor) {
        logsStringBuilder.append(sessionDescriptor)
    }

    fun logEndSession(timestamp: Long) {
        addLog(tag = "", msg = END_SESSION, timeStamp = timestamp, currentThread = "")
    }

    override fun interrupt() {
        wasInterrupted = true
        super.interrupt()
    }


}
