package com.instabug.library.instacapture.screenshot.pixelcopy

import android.app.Activity
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Paint
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.os.Build
import android.view.PixelCopy
import android.view.SurfaceView
import android.view.View
import android.view.ViewGroup
import androidx.annotation.RequiresApi
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.instacapture.screenshot.pixelcopy.PixelCopyHandlerProvider.handler
import com.instabug.library.util.InstabugSDKLogger
import io.reactivexport.Scheduler
import io.reactivexport.Single
import io.reactivexport.SingleEmitter
import io.reactivexport.SingleSource
import io.reactivexport.functions.Function
import io.reactivexport.schedulers.Schedulers

@RequiresApi(Build.VERSION_CODES.O)
fun appendSurfaceViewToScreenshotIfPresent(
    activity: Activity,
    processScheduler: Scheduler = Schedulers.io()
) =
    Function<Bitmap, SingleSource<Bitmap>> { fullScreenshot ->
        // Try to locate a SurfaceView in the activity's decorView.
        findSurfaceViewInHierarchy(activity.window.decorView)?.let { surfaceView ->
            captureSurfaceViewAndAppendToScreenshot(
                surfaceView,
                fullScreenshot, processScheduler
            )
        } ?: Single.just(fullScreenshot)
    }

@RequiresApi(Build.VERSION_CODES.O)
fun captureSurfaceViewAndAppendToScreenshot(
    surfaceView: SurfaceView,
    fullScreenshot: Bitmap,
    processScheduler: Scheduler
): Single<Bitmap> = Single.create { emitter ->
    try {
        val location = surfaceView.getScreenLocation()
        val surfaceScreenshot = Bitmap.createBitmap(
            surfaceView.width,
            surfaceView.height,
            fullScreenshot.config ?: Bitmap.Config.RGB_565// Use the same config as the full screenshot
        )
        emitter.onSuccess(surfaceScreenshot to location)
    } catch (t: Throwable) {
        // log error
        handleError(t, emitter) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Something went wrong while capturing surfaceView ${t.message.orEmpty()}",
                t
            )
        }
    }
}.observeOn(processScheduler)
    .flatMap { pair -> captureSurfaceViewBitmap(pair, surfaceView) }
    .map { pair ->
        compositeSurfaceViewOnScreenshot(pair, fullScreenshot)
        fullScreenshot
    }.onErrorReturnItem(fullScreenshot)

fun View.getScreenLocation(): IntArray {
    val location = IntArray(2)
    getLocationOnScreen(location)
    return location
}

@RequiresApi(Build.VERSION_CODES.O)
fun captureSurfaceViewBitmap(
    pair: Pair<Bitmap, IntArray>,
    surfaceView: SurfaceView
): Single<Pair<Bitmap, IntArray>> =
    Single.create { emitter: SingleEmitter<Pair<Bitmap, IntArray>> ->
        val (surfaceScreenshot, location) = pair
        try {
            requestSurfaceViewPixelCopy(
                surfaceView,
                surfaceScreenshot,
                createPixelCopyFinishedListener(emitter, surfaceScreenshot, location)
            )
        } catch (t: Throwable) {
            handleError(t, emitter) {
                IBGDiagnostics.reportNonFatal(
                    t,
                    "Error capturing SurfaceView: ${t.message.orEmpty()}"
                )
            }
        }
    }

fun compositeSurfaceViewOnScreenshot(
    pair: Pair<Bitmap, IntArray>,
    fullScreenshot: Bitmap
) {
    val (surfaceScreenshot, surfaceLocation) = pair
    val paint = Paint()
    paint.setXfermode(PorterDuffXfermode(PorterDuff.Mode.DST_ATOP))
    Canvas(fullScreenshot).drawBitmap(
        surfaceScreenshot,
        surfaceLocation[0].toFloat(),
        surfaceLocation[1].toFloat(),
        paint
    )
}

private inline fun handleError(
    t: Throwable,
    emitter: SingleEmitter<Pair<Bitmap, IntArray>>,
    execute: () -> Unit
) {
    execute()
    emitter.onError(t)
}

@RequiresApi(Build.VERSION_CODES.O)
fun requestSurfaceViewPixelCopy(
    surfaceView: SurfaceView,
    screenshot: Bitmap,
    listener: PixelCopy.OnPixelCopyFinishedListener
) {
    PixelCopy.request(
        surfaceView,
        screenshot,
        listener,
        handler
    )
}

fun findSurfaceViewInHierarchy(view: View?): SurfaceView? {
    return when (view) {
        is SurfaceView -> view
        is ViewGroup -> {
            var found: SurfaceView? = null
            for (i in 0 until view.childCount) {
                found = findSurfaceViewInHierarchy(view.getChildAt(i))
                if (found != null) break
            }
            found
        }
        else -> null
    }
}

@RequiresApi(Build.VERSION_CODES.O)
fun createPixelCopyFinishedListener(
    emitter: SingleEmitter<Pair<Bitmap, IntArray>>,
    screenshot: Bitmap,
    surfaceLocation: IntArray
): PixelCopy.OnPixelCopyFinishedListener = PixelCopy.OnPixelCopyFinishedListener { resultCode ->
    if (resultCode != PixelCopy.SUCCESS) {
        screenshot.recycle()
        val errorMsg = "Error capturing SurfaceView via PixelCopy.request, resultCode: $resultCode"
        InstabugSDKLogger.e(Constants.LOG_TAG, errorMsg)
        emitter.onError(Exception(errorMsg))
    } else {
        emitter.onSuccess(screenshot to surfaceLocation)
    }
}
