package com.instabug.library.featuresflags.caching

import android.database.Cursor
import androidx.annotation.WorkerThread
import com.instabug.library.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.featuresflags.di.FeaturesFlagServiceLocator
import com.instabug.library.featuresflags.model.IBGFeatureFlag
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.util.extenstions.runOrLogAndReport
import kotlin.math.roundToInt

class FeaturesFlagDBManagerImpl(private val dbManager: IBGDbManager?) : FeaturesFlagDBManager {
    @WorkerThread
    override fun insert(vararg ibgFeatureFlags: IBGFeatureFlag) {
        dbManager?.runCatching {
            val cachedFeatureFlags = getAll(1.0f)
            deleteCommonFeatureFlags(cachedFeatureFlags, ibgFeatureFlags.toList())
            val sqlString = insertIntoQuery(ibgFeatureFlags)
            this.execSQL(sqlString, convertForSqlBinding(ibgFeatureFlags))
        }?.runOrLogAndReport(
            "Inserting features flags to DataBase failed. ", tag = Constants.LOG_TAG
        )

    }

    private fun deleteCommonFeatureFlags(
        cachedFeatureFlags: List<IBGFeatureFlag>,
        newFeatureFlags: List<IBGFeatureFlag>
    ) {
        takeIf { cachedFeatureFlags.isNotEmpty() }?.let {
            val commonElements = newFeatureFlags.toSet().filter { newFeatureFlag ->
                cachedFeatureFlags.any { oldFeatureFlag ->
                    newFeatureFlag.key == oldFeatureFlag.key
                }
            }
            if (commonElements.isNotEmpty())
                delete(commonElements.map { it.key })
        }

    }

    @WorkerThread
    override fun delete(featureFlagKeys: List<String>) {
        val selection = "${IBGDbContract.FeaturesFlagsEntry.FEATURES_FLAGS_COLUMN_KEY} = ?"
        featureFlagKeys.toSet().forEach { featureKey ->
            val whereArgs = listOf(IBGWhereArg(featureKey, false))
            dbManager?.delete(IBGDbContract.FeaturesFlagsEntry.TABLE_NAME, selection, whereArgs)
        }
    }

    override fun getAll(percentage: Float): List<IBGFeatureFlag> {
        val featuresFlag = mutableListOf<IBGFeatureFlag>()
        var queriedCursor: Cursor? = null
        val desiredLimit =
            (FeaturesFlagServiceLocator.featuresFlagsConfigsProvider.storeLimit * percentage).roundToInt()
        try {
            queriedCursor = dbManager?.query(
                IBGDbContract.FeaturesFlagsEntry.TABLE_NAME, null, null, null, null, null, null,
                desiredLimit.toString()
            )
            queriedCursor?.let { cursor ->
                if (cursor.moveToFirst()) {
                    do {
                        val featureFlagKey =
                            cursor.getString(cursor.getColumnIndexOrThrow(IBGDbContract.FeaturesFlagsEntry.FEATURES_FLAGS_COLUMN_KEY))
                        val featureFlagValue =
                            cursor.getString(cursor.getColumnIndexOrThrow(IBGDbContract.FeaturesFlagsEntry.FEATURES_FLAGS_COLUMN_VALUE))
                        featuresFlag.add(IBGFeatureFlag(featureFlagKey, featureFlagValue))
                    } while (cursor.moveToNext())
                }
            }

        } catch (exception: Exception) {
            IBGDiagnostics.reportNonFatalAndLog(
                exception,
                "Error while getting Features flags from DB:  " + exception.message,
                Constants.LOG_TAG
            )
        } finally {
            queriedCursor?.close()
        }
        return featuresFlag
    }

    @WorkerThread
    override fun deleteAll() {
        dbManager?.delete(IBGDbContract.FeaturesFlagsEntry.TABLE_NAME, null, null)

    }

    @WorkerThread
    override fun trimToLimit(limit: Int): Int {
        var rowsDeleted = 0
        dbManager.runCatching {

            val subQuery = """
           SELECT ${IBGDbContract.FeaturesFlagsEntry.COLUMN_ID} 
           FROM ${IBGDbContract.FeaturesFlagsEntry.TABLE_NAME}
           ORDER BY ${IBGDbContract.FeaturesFlagsEntry.COLUMN_ID} DESC LIMIT ? OFFSET ?
        """.trimIndent()
            val whereClause = "${IBGDbContract.FeaturesFlagsEntry.COLUMN_ID} IN ($subQuery)"
            val whereArgs = mutableListOf(IBGWhereArg("-1", true), IBGWhereArg("$limit", true))
            rowsDeleted =
                this?.delete(IBGDbContract.FeaturesFlagsEntry.TABLE_NAME, whereClause, whereArgs) ?: 0
        }.runOrLogAndReport(
            "Delete features flags from DataBase when reached limit failed", tag = Constants.LOG_TAG
        )

        return rowsDeleted
    }


    //helper methods for interact with database layer
    private fun insertIntoQuery(featuresFlag: Array<out IBGFeatureFlag>): String {
        val placeHolders =
            featuresFlag.joinToString(separator = ", ", prefix = "", postfix = "") { "(?, ?)" }
        return """
        INSERT INTO ${IBGDbContract.FeaturesFlagsEntry.TABLE_NAME}
        (${IBGDbContract.FeaturesFlagsEntry.FEATURES_FLAGS_COLUMN_KEY},${IBGDbContract.FeaturesFlagsEntry.FEATURES_FLAGS_COLUMN_VALUE})
        Values $placeHolders
        """.trimIndent()

    }

    private fun convertForSqlBinding(featuresFlags: Array<out IBGFeatureFlag>): Array<String?> {
        return featuresFlags.flatMap { listOf(it.key, it.value) }.toTypedArray()
    }


}