package com.instabug.library.diagnostics.sdkEvents.cache

import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.COLUMN_COUNT
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.COLUMN_KEY
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.ColumnsTransitiveState.KEY_TRANSITIVE_STATE
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.INSERT_STATEMENT
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.TABLE_NAME
import com.instabug.library.diagnostics.diagnostics_db.SDKEventsEntry.UPDATE_STATEMENT
import com.instabug.library.diagnostics.sdkEvents.mappers.SDKEventDbMapper
import com.instabug.library.diagnostics.sdkEvents.models.SDKEvent
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.isSQLStatement
import com.instabug.library.util.extenstions.runOrLogError

internal class SDKEventsDbHelperImpl(
    private val sdkEventDbMapper: SDKEventDbMapper,
    private val dbManager: DiagnosticsDbManager?,
) : SDKEventsDbHelper {

    override fun insertOrUpdateEvent(event: SDKEvent) {
        event.takeUnless { it.key.isSQLStatement }
            ?.run {
                dbManager?.execSQL(
                    INSERT_STATEMENT.format(
                        event.key,
                        event.key,
                        event.count.toString()
                    )
                )
            }
    }

    override fun updateSyncedRecords(syncedEvents: List<SDKEvent>) {
        if (syncedEvents.isEmpty()) return

        syncedEvents.asSequence()
            .filterNot { it.key.isSQLStatement }
            .forEach {
                dbManager?.execSQL(
                    UPDATE_STATEMENT.format(
                        it.count.toString(),
                        it.count.toString(),
                        it.key
                    )
                )
            }
    }

    override fun deleteManyEvents(keys: Collection<String>?) {
        keys?.toList()
            ?.also { dbManager?.beginTransaction() }
            ?.onEach(::deleteByKey)
            ?.also { dbManager?.endTransaction() }
    }

    private fun deleteByKey(key: String) {
        dbManager?.delete(
            TABLE_NAME,
            "$COLUMN_KEY = ?",
            listOf(IBGWhereArg(key, KEY_TRANSITIVE_STATE))
        )
    }

    override fun deleteAllEvents() {
        dbManager?.delete(TABLE_NAME, null, null)
    }

    override fun queryAllEvents(): List<SDKEvent>? = dbManager
        ?.runOrLogError {
            query(TABLE_NAME, null, null, null, null, null, null)
                ?.use(sdkEventDbMapper::getSDKEventList)
        }
        ?.getOrNull()
        .also {
            InstabugSDKLogger.d(
                LOG_TAG,
                if (it.isNullOrEmpty()) "queryAllEvents $it"
                else "queryAllEvents $it"
            )
        }

    override fun deleteZeroCountedEvents() {
        dbManager?.delete(
            TABLE_NAME,
            "$COLUMN_COUNT = 0 ",
            null,
        )
    }
}