package com.instabug.library.diagnostics.nonfatals.cache;


import static com.instabug.library.diagnostics.diagnostics_db.NonFatalOccurrenceEntry.COLUMN_NON_FATAL_ID;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalOccurrenceEntry.COLUMN_REPORTED_AT;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalOccurrenceEntry.COLUMN_STATE_FILE;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalOccurrenceEntry.TABLE_NAME;
import static com.instabug.library.diagnostics.diagnostics_db.NonFatalOccurrenceEntry.TRIM_BY_LIMIT_WHERE_CLAUSE;

import android.database.Cursor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.diagnostics_db.DiagnosticsDbManager;
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator;
import com.instabug.library.diagnostics.nonfatals.model.Occurrence;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor;
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg;
import com.instabug.library.util.InstabugSDKLogger;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Shannan on 01/09/2021.
 */

public final class OccurrencesDBHelperImpl implements OccurrencesDBHelper {


    @Nullable
    private final DiagnosticsDbManager databaseManager = ServiceLocator.getDatabaseManager();

    @Override
    public boolean insertOccurrence(@NonNull Occurrence Occurrence) {
        if (databaseManager != null) {
            try {
                IBGContentValues contentValues = new IBGContentValues();
                contentValues.put(COLUMN_NON_FATAL_ID, Occurrence.getNonFatalId(), ColumnsTransitiveStates.NON_FATAL_ID);
                if (Occurrence.getStateFileUri() != null) {
                    contentValues.put(COLUMN_STATE_FILE, Occurrence.getStateFileUri(), ColumnsTransitiveStates.STATE_FILE);
                }
                contentValues.put(COLUMN_REPORTED_AT, Occurrence.getReportedAt(), ColumnsTransitiveStates.REPORTED_AT);
                long id = databaseManager.insertWithOnConflict(TABLE_NAME, null, contentValues);
                return id != -1;
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while inserting non-fatal occurrence", e);
            }
        }
        return false;
    }

    @Override
    public void trimToLimit(int limit) {
        if (databaseManager != null) {
            try {

                List<IBGWhereArg> whereArgs = new ArrayList<>();
                whereArgs.add(new IBGWhereArg("-1", true));
                whereArgs.add(new IBGWhereArg(String.valueOf(limit), true));
                databaseManager.delete(TABLE_NAME, TRIM_BY_LIMIT_WHERE_CLAUSE, whereArgs);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while trimming non-fatal table", e);
            }
        }
    }

    @NotNull
    @Override
    public List<Occurrence> getAllOccurrences() {
        List<Occurrence> occurrences = new ArrayList<>();
        if (databaseManager != null) {
            IBGCursor cursor = null;
            try {
                cursor = databaseManager.query(TABLE_NAME, null, null, null, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        Occurrence occurrence = new Occurrence(cursor.getLong(cursor.getColumnIndex(COLUMN_NON_FATAL_ID)),
                                cursor.getLong(cursor.getColumnIndex(COLUMN_REPORTED_AT)),
                                cursor.getString(cursor.getColumnIndex(COLUMN_STATE_FILE)));
                        occurrences.add(occurrence);
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving occurrences", e);
            } finally {
                if (cursor != null) {
                    try {
                        cursor.close();
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Cursor not closed", e);
                    }
                }
            }
        }

        return occurrences;
    }


    @Override
    public int getNonFatalOccurrencesCount(long nonFatalId) {
        if (databaseManager != null) {
            String selectionClause = COLUMN_NON_FATAL_ID + " = ?";
            List<IBGWhereArg> selectionArgs = new ArrayList<>();
            selectionArgs.add(new IBGWhereArg(String.valueOf(nonFatalId), ColumnsTransitiveStates.NON_FATAL_ID));
            IBGCursor cursor = null;
            try {
                cursor = databaseManager.query(TABLE_NAME, null, selectionClause, selectionArgs, null,null,null);
                if (cursor != null && cursor.moveToFirst()) {
                    return cursor.getCount();
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving occurrences count", e);
            } finally {
                if (cursor != null) {
                    try {
                        cursor.close();
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Cursor not closed", e);
                    }
                }
            }
        }
        return NOT_FOUND;
    }

    @NotNull
    @Override
    public List<Occurrence> getNonFatalOccurrences(long nonFatalId) {
        List<Occurrence> occurrences = new ArrayList<>();
        if (databaseManager != null) {
            String selectionClause = COLUMN_NON_FATAL_ID + " = ?";
            List<IBGWhereArg> selectionArgs = new ArrayList<>();
            selectionArgs.add(new IBGWhereArg(String.valueOf(nonFatalId), ColumnsTransitiveStates.NON_FATAL_ID));
            IBGCursor cursor = null;
            try {
                cursor = databaseManager.query(TABLE_NAME, null, selectionClause, selectionArgs, null, null, null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        Occurrence occurrence = new Occurrence(cursor.getLong(cursor.getColumnIndex(COLUMN_NON_FATAL_ID)),
                                cursor.getLong(cursor.getColumnIndex(COLUMN_REPORTED_AT)),
                                cursor.getString(cursor.getColumnIndex(COLUMN_STATE_FILE)));
                        occurrences.add(occurrence);
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while retrieving occurrences", e);
            } finally {
                if (cursor != null) {
                    try {
                        cursor.close();
                    } catch (Exception e) {
                        InstabugSDKLogger.e(Constants.LOG_TAG, "Cursor not closed", e);
                    }
                }
            }
        }

        return occurrences;
    }

    @Override
    public void update(Occurrence occurrence) {
        if (databaseManager != null) {
            try {
                IBGContentValues contentValues = new IBGContentValues();
                contentValues.put(COLUMN_NON_FATAL_ID, occurrence.getNonFatalId(), ColumnsTransitiveStates.NON_FATAL_ID);
                if (occurrence.getStateFileUri() != null)
                    contentValues.put(COLUMN_STATE_FILE, occurrence.getStateFileUri(), ColumnsTransitiveStates.STATE_FILE);
                List<IBGWhereArg> selectionArgs = new ArrayList<>();
                selectionArgs.add(new IBGWhereArg(String.valueOf(occurrence.getReportedAt()), ColumnsTransitiveStates.REPORTED_AT));
                databaseManager.update(TABLE_NAME,
                        contentValues,
                        COLUMN_REPORTED_AT + " = ?",
                        selectionArgs);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while updating occurrence", e);
            }
        }
    }

    @Override
    public void clearNonFatalOccurrences(int nonFatalId) {
        if (databaseManager != null) {
            try {
                List<IBGWhereArg> whereArgs = new ArrayList<>();
                whereArgs.add(new IBGWhereArg(String.valueOf(nonFatalId), ColumnsTransitiveStates.NON_FATAL_ID));
                databaseManager.delete(TABLE_NAME, COLUMN_NON_FATAL_ID + " = ?", whereArgs);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while deleting occurrences ", e);
            }
        }
    }

    @Override
    public void clearAll() {
        if (databaseManager != null) {
            try {
                databaseManager.delete(TABLE_NAME, null, null);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while clearing occurrences", e);
            }
        }
    }

    @Nullable
    @Override
    public String[] getStateFilesForNonFatal(long id) {
        if (databaseManager != null) {
            Cursor cursor = null;
            try {
                List<IBGWhereArg> selectionArguments = new ArrayList<>();
                selectionArguments.add(new IBGWhereArg(String.valueOf(id), ColumnsTransitiveStates.NON_FATAL_ID));
                cursor = databaseManager.query(TABLE_NAME,
                        new String[]{COLUMN_STATE_FILE},
                        COLUMN_NON_FATAL_ID + " = ?",
                        selectionArguments,
                        null,
                        null,
                        null);
                String[] stateFiles = null;
                if (cursor != null && cursor.moveToFirst()) {
                    stateFiles = new String[cursor.getCount()];
                    int currentIndex = 0;
                    do {
                        stateFiles[currentIndex] = cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_STATE_FILE));
                        currentIndex++;
                    } while (cursor.moveToNext());
                }
                if (cursor != null) {
                    cursor.close();
                }
                return stateFiles;
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while getting non fatal state files", e);
            } finally {
                if (cursor != null && !cursor.isClosed()) {
                    cursor.close();
                }
            }
        }
        return null;
    }

    @Override
    public List<String> getStateFiles() {
        List<String> stateFiles = new ArrayList<>();
        if (databaseManager != null) {
            Cursor cursor = null;
            try {
                List<IBGWhereArg> selectionArguments = new ArrayList<>();
                cursor = databaseManager.query(TABLE_NAME,
                        new String[]{COLUMN_STATE_FILE},
                        null,
                        selectionArguments,
                        null,
                        null,
                        null);
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        stateFiles.add(cursor.getString(cursor.getColumnIndexOrThrow(COLUMN_STATE_FILE)));
                    } while (cursor.moveToNext());
                }
                if (cursor != null) {
                    cursor.close();
                }
                return stateFiles;
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while getting non fatal state files", e);
            } finally {
                if (cursor != null && !cursor.isClosed()) {
                    cursor.close();
                }
            }
        }

        return stateFiles;
    }

    @Override
    public void deleteOccurrence(String stateFileUri) {
        if (databaseManager != null) {
            try {
                String whereClause = COLUMN_STATE_FILE + " = ?";
                List<IBGWhereArg> args = new ArrayList<>();
                args.add(new IBGWhereArg(String.valueOf(stateFileUri), ColumnsTransitiveStates.STATE_FILE));
                databaseManager.delete(TABLE_NAME, whereClause, args);
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while deleting non-fatals", e);
            }
        }
    }
}
