package com.instabug.library.core.ui;

import android.annotation.SuppressLint;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.CallSuper;
import androidx.annotation.IdRes;
import androidx.annotation.LayoutRes;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.annotation.VisibleForTesting;
import androidx.fragment.app.Fragment;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.util.LocaleUtils;

import org.jetbrains.annotations.NotNull;


/**
 * Created by tarek on 9/25/16.
 */
public abstract class InstabugBaseFragment<P extends BaseContract.Presenter> extends Fragment
        implements BaseContract.View<Fragment> {

    @Nullable
    protected P presenter;
    protected View rootView;

    @Nullable
    protected IBGTheme theme = null;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
    }

    @CallSuper
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        rootView = inflater.inflate(getLayout(), container, false);
        if (presenter != null) {
            theme = presenter.getTheme();
        }
        initViews(rootView, savedInstanceState);
        return rootView;
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
    }

    @Override
    public void onStart() {
        super.onStart();
    }

    @Override
    public void onStop() {
        super.onStop();
    }

    @Override
    public void onDestroyView() {
        rootView = null;
        super.onDestroyView();
    }

    @Override
    public Fragment getViewContext() {
        return this;
    }

    @Override
    public void finishActivity() {
        if (getActivity() != null) {
            getActivity().finish();
        }
    }

    /**
     * Look for a child view with the given id.  If this view has the given
     * id, return this view.
     *
     * @param id The id to search for.
     * @return The view that has the given id in the hierarchy or null
     */
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    protected <T extends View> T findViewById(@IdRes int id) {
        if (rootView == null) return null;

        return rootView.findViewById(id);
    }

    /**
     * @return layout resource id.
     */
    protected abstract @LayoutRes
    int getLayout();

    /**
     * Init fragment views
     *
     * @param rootView           fragment root view.
     * @param savedInstanceState saved instance bundle
     */
    protected abstract void initViews(View rootView, @Nullable Bundle savedInstanceState);

    protected String getLocalizedString(@StringRes int resourceId) {
        return LocaleUtils.getLocaleStringResource(
                InstabugCore.getLocale(getContext()),
                resourceId,
                getContext());
    }

    protected String getLocalizedString(@StringRes int resourceId, Object... formatArgs) {
        return LocaleUtils.getLocaleStringResource(
                InstabugCore.getLocale(getContext()),
                resourceId,
                getContext(),
                formatArgs);
    }


    @VisibleForTesting
    public void setTheme(IBGTheme theme) {
        this.theme = theme;
    }
}
