package com.instabug.library;

import android.app.Activity;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

import java.lang.ref.WeakReference;
import java.util.ArrayList;

/**
 * This class responsible for showing activities that auto-show like Surveys & OnBoarding
 */
public class PresentationManager {
    @Nullable
    private static volatile PresentationManager INSTANCE;
    @NonNull
    private final ArrayList<Runnable> screensList;
    @Nullable
    private WeakReference<Activity> currentActivity;
    @Nullable
    private String lastRunnableName;
    private boolean isNotificationShowing = false;
    private boolean isInInstabugContext = false;

    private PresentationManager() {
        screensList = new ArrayList<>();
    }

    public synchronized static PresentationManager getInstance() {
        return INSTANCE != null ? INSTANCE : (INSTANCE = new PresentationManager());
    }

    public void show(Runnable runnable) {
        synchronized (screensList) {
            if (runnable == null || hasPreviousRunnable(screensList))
                return;
            screensList.add(runnable);
            InstabugSDKLogger.v(Constants.LOG_TAG, "screensList Size:" + screensList.size());
            InstabugSDKLogger.v(Constants.LOG_TAG, "currentActivity != null:" + (currentActivity != null));
            InstabugSDKLogger.v(Constants.LOG_TAG, "currentActivity.get() != null:" +
                    (currentActivity != null && currentActivity.get() != null ? currentActivity :
                            " activity isn't provided"));
            InstabugSDKLogger.v(Constants.LOG_TAG, "!(currentActivity.get() instanceof _InstabugActivity:" +
                    " " + (currentActivity != null && !(currentActivity.get() instanceof _InstabugActivity)));
            InstabugSDKLogger.v(Constants.LOG_TAG, "!isNotificationShowing:" + !isNotificationShowing);
            InstabugSDKLogger.v(Constants.LOG_TAG, "!isInInstabugContext:" + (!isInInstabugContext));
            if (screensList.size() == 1
                    && instabugNotDisplayed()
                    && !isNotificationShowing
                    && !isInInstabugContext) {
                notifyActivityChanged();
            }
        }
    }

    private boolean instabugNotDisplayed() {
        if (currentActivity == null || currentActivity.get() == null) {
            return !InstabugCore.isForegroundBusy();
        } else {
            return !(currentActivity.get() instanceof _InstabugActivity);
        }
    }

    /*
       Avoid adding duplicated runnables in case of surveys for example.
     */
    private boolean hasPreviousRunnable(ArrayList<Runnable> screensList) {
        for (Runnable runnable : screensList) {
            if (runnable.getClass().getName().equals(lastRunnableName)) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "hasPreviousRunnable");
                return true;
            }
        }
        InstabugSDKLogger.v(Constants.LOG_TAG, "hasPreviousRunnable false");
        return false;
    }

    public synchronized void notifyActivityChanged() {
        PoolProvider.postIOTaskWithCheck(() -> {
            boolean foregroundNotBusy = InstabugCore.isForegroundNotBusy();
            InstabugSDKLogger.v(Constants.LOG_TAG, "screensList.size(): " + (screensList.size()));
            InstabugSDKLogger.v(Constants.LOG_TAG, "InstabugCore.foregroundNotBusy(): " + (foregroundNotBusy));
            if (!screensList.isEmpty() && foregroundNotBusy && Instabug.isEnabled()) {
                lastRunnableName = screensList.get(0).getClass().getName();
                Runnable runnable = screensList.remove(0);
                PoolProvider.postMainThreadTask(runnable);
            }
        });
    }

    public void setCurrentActivity(Activity activity) {
        currentActivity = new WeakReference<>(activity);
    }

    public String getCurrentActivityName() {
        if (currentActivity != null) {
            Activity activity = currentActivity.get();
            if (activity != null) {
                return activity.getLocalClassName();
            }
        }
        return "";
    }

    public void setNotificationShowing(boolean isNotificationShowing) {
        this.isNotificationShowing = isNotificationShowing;
    }

    public void setInInstabugContext(boolean inInstabugContext) {
        isInInstabugContext = inInstabugContext;
    }

    public static void release() {
        INSTANCE = null;
    }
}