package com.instabug.library.sessionreplay.monitoring

import com.instabug.library.internal.filestore.DataAggregator
import com.instabug.library.internal.filestore.IDsMatchMultiSpanSelector
import com.instabug.library.map.Mapper
import com.instabug.library.model.v3Session.IBGSessionData
import com.instabug.library.sessionV3.providers.FeatureSessionDataController
import com.instabug.library.sessionV3.providers.FeatureSessionDataControllerHost
import com.instabug.library.sessionV3.providers.SessionID
import com.instabug.library.sessionreplay.SR_LOG_TAG
import com.instabug.library.sessionreplay.monitoring.AnalyticsSessionDataMapper.SessionKeys.SRAnalyticsKey
import com.instabug.library.util.extenstions.logVWithThreadName
import org.json.JSONArray
import org.json.JSONObject

class AnalyticsSessionDataMapper : Mapper<SRAnalytics, JSONObject> {

    override fun map(from: SRAnalytics): JSONObject = JSONObject().apply {
        getReasonsList(from).takeUnless { it.length() == 0 }
            ?.let { put(DroppingReasons, it) }
        put(IBGLogsCount, from.ibgLogsCount)
        put(NetworkLogsCount, from.networkLogsCount)
        put(UserStepsCount, from.userStepsCount)
        put(ScreenshotsMetadataCount, from.screenshotsMetadataCount)
        val totalLogsCount =
            with(from) { ibgLogsCount + networkLogsCount + userStepsCount + screenshotsMetadataCount }
        put(TotalLogsCount, totalLogsCount)
        put(ScreenshotsCount, from.screenshotsCount)
        put(SamplingDrops, from.samplingDrops)
        put(SessionStorageViolationDrops, from.sessionStorageViolationDrops)
        put(ScreenshotsSessionStorageViolationDrops, from.screenshotsStorageViolationDrops)
        put(Errors, JSONArray(from.errors))
    }

    private fun getReasonsList(input: SRAnalytics): JSONArray = JSONArray().apply {
        input.samplingDrops.takeIf { it > 0 }
            ?.also { put(SamplingDroppingReason) }
        input.sessionStorageViolationDrops.takeIf { it > 0 }
            ?.also { put(SessionSizeDroppingReason) }
        input.aggregateStorageViolation.takeIf { it }
            ?.also { put(AggregateSizeDroppingReason) }
        input.screenshotsStorageViolationDrops.takeIf { it > 0 }
            ?.also { put(ScreenshotsSessionSizeDroppingReason) }
        takeIf { input.isSDKSampled }
            ?.also { put(SessionReplaySDKSampled) }
    }

    companion object SessionKeys {
        // region Session Request Keys
        const val SRAnalyticsKey = "sra"
        const val DroppingReasons = "dd"
        const val IBGLogsCount = "il"
        const val NetworkLogsCount = "nl"
        const val UserStepsCount = "ul"
        const val ScreenshotsMetadataCount = "sml"
        const val TotalLogsCount = "t"
        const val ScreenshotsCount = "st"
        const val SamplingDrops = "s"
        const val SessionStorageViolationDrops = "ss"
        const val ScreenshotsSessionStorageViolationDrops = "sss"
        const val Errors = "e"
        // endregion

        // region Dropping reasons
        const val SamplingDroppingReason = "Sampling"
        const val SessionSizeDroppingReason = "SessionSize"
        const val AggregateSizeDroppingReason = "DiskSize"
        const val ScreenshotsSessionSizeDroppingReason = "ScreenshotsSize"
        const val SessionReplaySDKSampled = "SDKSampled"
        // endregion
    }
}

class SRAnalyticsDataAggregator : DataAggregator<SRAnalytics?> {
    var current: JSONObject? = null
    override fun add(data: JSONObject) {
        current = data
    }

    override fun aggregate(): SRAnalytics? = current?.let(SRAnalytics.Helper::fromJson)
}

class SRSessionDataController(
    private val dataStore: SRMonitoringSpansDataStore,
    private val dataMapper: Mapper<SRAnalytics, JSONObject>
) : FeatureSessionDataController {
    override fun collectSessionsData(sessionsIds: List<SessionID>): Map<SessionID, IBGSessionData> {
        "[Monitoring] Collecting session data for sessions $sessionsIds"
            .logVWithThreadName(SR_LOG_TAG)
        return dataStore.retrieve(
            aggregator = SRAnalyticsDataAggregator(),
            spansSelector = IDsMatchMultiSpanSelector(sessionsIds)
        ).get().filterNotNull()
            .associate { it.sessionId to IBGSessionData(SRAnalyticsKey, dataMapper.map(it)) }
    }

    override fun dropSessionData(sessionsIds: List<SessionID>) {
        "[Monitoring] Dropping session data for sessions $sessionsIds"
            .logVWithThreadName(SR_LOG_TAG)
        dataStore.delete(IDsMatchMultiSpanSelector(sessionsIds))
    }
}

class SRSessionDataControllerHost(
    override val sessionDataController: FeatureSessionDataController
) : FeatureSessionDataControllerHost