package com.instabug.library.sessionprofiler

import android.annotation.SuppressLint
import android.app.ActivityManager
import android.app.ActivityManager.MemoryInfo
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.Configuration
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.BatteryManager
import android.os.Build
import android.telephony.TelephonyManager
import androidx.annotation.RequiresApi
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.Constants
import com.instabug.library.Instabug
import com.instabug.library.sessionprofiler.model.timeline.BatteryState
import com.instabug.library.sessionprofiler.model.timeline.ConnectivityState
import com.instabug.library.sessionprofiler.model.timeline.MemoryUsage
import com.instabug.library.sessionprofiler.model.timeline.ScreenOrientationMode
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.nullRetryLazy
import com.instabug.library.util.weakRetryLazy

class AttributesProvider {

    private val applicationContext by nullRetryLazy { Instabug.getApplicationContext() }

    @get:Throws(NullContextException::class)
    private val activityManager by weakRetryLazy {
        requireContext().getSystemService(Context.ACTIVITY_SERVICE) as? ActivityManager
    }

    @get:Throws(NullContextException::class)
    private val externalCacheDir by weakRetryLazy { requireContext().externalCacheDir }

    @get:Throws(NullContextException::class)
    private val connectivityManager by weakRetryLazy {
        requireContext().getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
    }

    @get:Throws(NullContextException::class)
    private val telephonyManager by weakRetryLazy {
        requireContext().getSystemService(Context.TELEPHONY_SERVICE) as? TelephonyManager
    }

    val batteryState: BatteryState
        @Throws(NullContextException::class)
        get() {
            try {
                val battery = IntentFilter(Intent.ACTION_BATTERY_CHANGED)
                val currentBattery = requireContext().registerReceiver(null, battery)
                if (currentBattery != null) {
                    return BatteryState(getBatteryLevel(currentBattery), isPlugged(currentBattery))
                }
                InstabugSDKLogger.d(Constants.LOG_TAG, "Couldn't obtain battery status")
            } catch (e: NullContextException) {
                throw e
            } catch (e: Exception) {
                InstabugSDKLogger.e(
                    Constants.LOG_TAG,
                    "Got error while obtaining battery status",
                    e
                )
            }
            return BatteryState(DEFAULT_FLOAT, true)
        }

    val screenOrientation: ScreenOrientationMode
        @Throws(NullContextException::class)
        get() = requireContext()
            .takeIf { ctx -> ctx.isInLandscape() }
            ?.let { ScreenOrientationMode.forLandscape() } ?: ScreenOrientationMode.forPortrait()

    val connectivityState: ConnectivityState
        @Throws(NullContextException::class)
        get() = ConnectivityState()
            .apply { value = ConnectivityState.KEY_NO_CONNECTION }
            .apply(::getNetworkStateCompat)

    private val carrier: String
        get() = runCatching {
            telephonyManager?.networkOperatorName
        }.getOrDefault(DEFAULT_CARRIER) ?: DEFAULT_CARRIER

    val memoryUsage: MemoryUsage
        @Throws(NullContextException::class)
        get() = activityManager?.let { manager ->
            val memoryInfo = MemoryInfo()
            manager.getMemoryInfo(memoryInfo)
            val totalMemory = memoryInfo.totalMem
            val usedMemory = totalMemory - memoryInfo.availMem
            MemoryUsage(usedMemory.toMegabyte(), totalMemory.toMegabyte())
        } ?: MemoryUsage(DEFAULT_LONG, DEFAULT_LONG)

    val storageUsage: MemoryUsage
        @Throws(NullContextException::class)
        get() = externalCacheDir?.takeIf { it.exists() }?.let { cacheDir ->
            val usedStorage = cacheDir.totalSpace - cacheDir.freeSpace
            MemoryUsage(usedStorage.toMegabyte())
        } ?: MemoryUsage(DEFAULT_LONG)

    private fun getBatteryLevel(currentBattery: Intent): Float {
        try {
            val level = currentBattery.getIntExtra(BatteryManager.EXTRA_LEVEL, DEFAULT_INT)
            val scale = currentBattery.getIntExtra(BatteryManager.EXTRA_SCALE, DEFAULT_INT)
            val batteryPct = level / scale.toFloat()
            return batteryPct.fromPct()
        } catch (e: Exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while obtaining battery level", e)
        }
        return DEFAULT_FLOAT
    }

    private fun isPlugged(currentBattery: Intent): Boolean {
        try {
            val status = currentBattery.getIntExtra(BatteryManager.EXTRA_STATUS, DEFAULT_INT)
            val isCharging = status == BatteryManager.BATTERY_STATUS_CHARGING ||
                    status == BatteryManager.BATTERY_STATUS_FULL
            return isCharging
        } catch (e: Exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Got error while obtaining battery state", e)
        }
        return true
    }

    private inline fun Context.isInLandscape() =
        resources.configuration.orientation == Configuration.ORIENTATION_LANDSCAPE

    // API check is suppressed to be able to use BuildFieldsProvider
    @SuppressLint("NewApi")
    @Throws(NullContextException::class)
    private fun getNetworkStateCompat(state: ConnectivityState) {
        connectivityManager?.let { manager ->
            if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.Q) {
                getNetworkStateAPI29(manager, state)
            } else {
                getNetworkStatePreAPI29(manager, state)
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.Q)
    private fun getNetworkStateAPI29(manager: ConnectivityManager, state: ConnectivityState) {
        manager.run { getNetworkCapabilities(activeNetwork) }
            ?.let { capabilities ->
                state.value =
                    if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)) {
                        ConnectivityState.KEY_CELLULAR
                    } else if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                        ConnectivityState.KEY_WIFI
                    } else {
                        ConnectivityState.KEY_NO_CONNECTION
                    }
            }
    }

    @Suppress("DEPRECATION")
    private fun getNetworkStatePreAPI29(manager: ConnectivityManager, state: ConnectivityState) {
        manager.activeNetworkInfo?.let { networkInfo ->
            state.value = when (networkInfo.type) {
                ConnectivityManager.TYPE_WIFI -> ConnectivityState.KEY_WIFI
                ConnectivityManager.TYPE_MOBILE -> {
                    // connected to the mobile provider's data plan
                    state.name = carrier
                    networkInfo.subtypeName
                }

                else -> ConnectivityState.KEY_NO_CONNECTION
            }
        }
    }

    class NullContextException : Exception()

    private inline fun requireContext(): Context =
        applicationContext ?: throw NullContextException()

    companion object {
        private const val SIZE_UNIT = 1024
        private const val PCT_UNIT = 100
        private const val DEFAULT_INT = -1
        private const val DEFAULT_LONG = -1L
        private const val DEFAULT_FLOAT = -1f
        private const val DEFAULT_CARRIER = "unknown"
        private fun Long.toMegabyte() = this / (SIZE_UNIT * SIZE_UNIT)
        private fun Float.fromPct() = this * PCT_UNIT
    }
}
