package com.instabug.library.sessionV3.manager

import com.instabug.library.Constants
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.model.v3Session.Defaults.DEFAULT_LAST_FOREGROUND_TIME
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STITCHING_ENABLED
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STITCHING_SESSION_TIMEOUT
import com.instabug.library.model.v3Session.PreferencesKeys.LAST_FOREGROUND_TIME_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.STITCHING_ENABLED_KEY
import com.instabug.library.model.v3Session.PreferencesKeys.STITCHING_SESSION_TIMEOUT_KEY
import com.instabug.library.util.InstabugSDKLogger
import java.util.concurrent.TimeUnit

interface SessionStitchingManger {
    var isStitchingEnabled: Boolean
    var sessionTimeoutInSeconds: Int
    var lastForegroundMicroTime: Long
    fun isSessionLead(sessionForegroundStartTime: Long): Boolean
}

object SessionStitchingMangerImpl : SessionStitchingManger {
    override var isStitchingEnabled: Boolean
            by corePref(
                key = STITCHING_ENABLED_KEY,
                defaultValue = DEFAULT_STITCHING_ENABLED
            )
    override var lastForegroundMicroTime: Long
            by corePref(
                key = LAST_FOREGROUND_TIME_KEY,
                defaultValue = DEFAULT_LAST_FOREGROUND_TIME
            )
    override var sessionTimeoutInSeconds: Int
            by corePref(
                key = STITCHING_SESSION_TIMEOUT_KEY,
                defaultValue = DEFAULT_STITCHING_SESSION_TIMEOUT
            )

    override fun isSessionLead(sessionForegroundStartTime: Long):Boolean {
        if (!isStitchingEnabled) return false
        val timeInBackgroundInSeconds = getTimeInBackgroundInSeconds(sessionForegroundStartTime)
        return if (timeInBackgroundInSeconds == DEFAULT_LAST_FOREGROUND_TIME || timeInBackgroundInSeconds> sessionTimeoutInSeconds) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "started new billable session")
            true
        } else {
            InstabugSDKLogger.v(Constants.LOG_TAG, "session stitched")
            false
        }
    }

    private fun getTimeInBackgroundInSeconds(sessionForegroundStartTime: Long): Long {
        return if (lastForegroundMicroTime != DEFAULT_LAST_FOREGROUND_TIME) {
            TimeUnit.MICROSECONDS.toSeconds(sessionForegroundStartTime - lastForegroundMicroTime)
        } else lastForegroundMicroTime
    }


}