package com.instabug.library.networkv2.request;

import static com.instabug.library.networkv2.request.Constants.BASE_URL;
import static com.instabug.library.networkv2.request.Header.ANDROID_VERSION;
import static com.instabug.library.networkv2.request.Header.APP_VARIANT;
import static com.instabug.library.networkv2.request.Header.SDK_VERSION;
import static com.instabug.library.networkv2.request.Header.SYSTEM_PLATFORM_OS;

import android.os.Build;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tokenmapping.TokenMappingServiceLocator;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Abstract class representing any API request.
 */
public class Request {

    private static final String SHORTEN_APP_TOKEN = "at";
    private static final String SHORTEN_UUID = "uid";
    private static final String APP_TOKEN = "application_token";
    private static final String UUID = "uuid";
    public static final String BASIC_AUTH_VALUE_PREFIX = "Basic ";

    @Nullable
    @VisibleForTesting
    String appTokenValue = null;
    private final String uuidValue = UserManager.getUUID();
    private final String sdkVersionValue = DeviceStateProvider.getSdkVersion();

    private final String requestUrl;
    @Nullable
    private final String endPoint;
    @Nullable
    @RequestMethod
    private final String requestMethod;
    @RequestType
    private final int requestType;
    private final List<RequestParameter> urlParameters;
    private final List<RequestParameter> bodyParameters;
    private final List<RequestParameter<String>> headers;
    @Nullable
    private final FileToUpload fileToUpload;
    @Nullable
    private final File downloadedFile;

    @Nullable
    private final JSONObject requestBodyJsonObject;
    private final boolean shorten;
    private boolean hasUuid = true;
    private boolean noDefaultParameters = false;

    public Request(Builder builder) {
        this.endPoint = builder.endpoint;
        this.requestUrl = builder.requestUrl != null ? builder.requestUrl : BASE_URL + endPoint;
        this.requestType = builder.type != RequestType.UNDEFINED ? builder.type : RequestType.NORMAL;
        this.requestMethod = builder.method;
        this.fileToUpload = builder.fileToUpload;
        this.downloadedFile = builder.downloadedFile;
        this.requestBodyJsonObject = builder.requestBodyJsonObject;
        this.shorten = builder.shorten;
        this.urlParameters = builder.urlParameters != null ? builder.urlParameters : new ArrayList<RequestParameter>();
        this.bodyParameters = builder.bodyParameters != null ? builder.bodyParameters : new ArrayList<RequestParameter>();
        this.headers = builder.headers != null ? builder.headers : new ArrayList<RequestParameter<String>>();
        this.hasUuid = builder.hasUuid;
        this.noDefaultParameters = builder.noDefaultParameters;
        this.appTokenValue = builder.tokenProvider.getAppToken();
        handleBaseParams(shorten, hasUuid, noDefaultParameters);
    }

    /**
     * Add the common param that are existing in every request.
     * Handle the case of unidentified (anonymous) requests@param identified determine if the UUID is required in the request or not.
     *
     * @param shorten determine if the base param keys is shorten
     */
    private void handleBaseParams(boolean shorten, boolean hasUuid, boolean noDefaultParameters) {
        // Add base params
        this.headers.add(new RequestParameter<>(SDK_VERSION, sdkVersionValue));

        if (noDefaultParameters) {
            return;
        } else if (!shorten) {
            if (appTokenValue != null) {
                addParameter(new RequestParameter<>(APP_TOKEN, appTokenValue));
            }

            if (hasUuid) {
                addParameter(new RequestParameter<>(UUID, uuidValue));
            }
        } else {
            if (appTokenValue != null) {
                addParameter(new RequestParameter<>(SHORTEN_APP_TOKEN, appTokenValue));
            }

            if (hasUuid) {
                addParameter(new RequestParameter<>(SHORTEN_UUID, uuidValue));
            }
        }
    }

    /**
     * Gets endpoint.
     *
     * @return the endpoint string
     */
    @Nullable
    public String getEndpoint() {
        return endPoint;
    }

    /**
     * Get request url.
     *
     * @return the string
     */
    public String getRequestUrl() {
        if (!getUrlEncodedParameters().isEmpty()) {
            return requestUrl + getUrlEncodedParameters();
        }
        return requestUrl;
    }

    /**
     * Get request url for logging purposes.
     *
     * @return if the {@link SettingsManager#shouldLogExtraRequestData()} is enabled
     * returns the full url with its query parameters otherwise it returns the url only.
     */
    public String getRequestUrlForLogging() {
        if (SettingsManager.shouldLogExtraRequestData() && !getUrlEncodedParameters().isEmpty()) {
            return requestUrl + getUrlEncodedParameters();
        }
        return requestUrl;
    }

    /**
     * get request method.
     *
     * @return request method {@link RequestMethod}
     */
    @RequestMethod
    public String getRequestMethod() {
        if (requestMethod == null) {
            return RequestMethod.GET;
        }
        return requestMethod;
    }

    /**
     * get request type.
     *
     * @return request type {@link RequestType}
     */
    public @RequestType
    int getRequestType() {
        return requestType;
    }

    /**
     * get request headers.
     *
     * @return request headers as a list of {@link RequestParameter}
     */
    public List<RequestParameter<String>> getHeaders() {
        return Collections.unmodifiableList(headers);
    }

    /**
     * get request url parameters.
     *
     * @return request parameters as a list of {@link RequestParameter}
     */
    public List<RequestParameter> getRequestUrlParameters() {
        return Collections.unmodifiableList(urlParameters);
    }

    /**
     * get request parameters.
     *
     * @return request parameters as a list of {@link RequestParameter}
     */
    public List<RequestParameter> getRequestBodyParameters() {
        return Collections.unmodifiableList(bodyParameters);
    }

    /**
     * Get url encoded parameters.
     *
     * @return Request parameters as url encoded string.
     */
    private String getUrlEncodedParameters() {
        UriWrapper.Builder builder = UriWrapper.Builder.getInstance();
        for (RequestParameter requestParameter : urlParameters) {
            builder.appendQueryParameter(requestParameter.getKey(), requestParameter.getValue().toString());
        }
        return builder.toString();
    }

    /**
     * Add url parameter {@link RequestParameter} to the request if the {@link RequestMethod} is GET or DELETE, otherwise,
     * add body parameter
     */
    private void addParameter(RequestParameter requestParameter) {
        if (requestMethod != null) {
            if (requestMethod.equals(RequestMethod.GET) || requestMethod.equals(RequestMethod.DELETE)) {
                addUrlParameter(requestParameter);
            } else {
                addBodyParameter(requestParameter);
            }
        }
    }

    /**
     * Add url parameter {@link RequestParameter} to the request if the {@link RequestMethod} is GET or DELETE, otherwise,
     * add body parameter
     */
    private void addUrlParameter(RequestParameter parameter) {
        urlParameters.add(parameter);
    }

    /**
     * Add body parameter {@link RequestParameter} to the request
     */
    private void addBodyParameter(RequestParameter parameter) {
        bodyParameters.add(parameter);
    }

    /**
     * Get request body.
     *
     * @return request body as string.
     */
    public String getRequestBody() {
        final @NonNull JSONObject bodyJsonObject;
        if (requestBodyJsonObject == null) {
            bodyJsonObject = new JSONObject();
        } else {
            bodyJsonObject = requestBodyJsonObject;
        }
        try {
            for (RequestParameter requestParameter : getRequestBodyParameters()) {
                bodyJsonObject.put(requestParameter.getKey(), requestParameter.getValue());
            }
            return bodyJsonObject.toString();
        } catch (JSONException | OutOfMemoryError e) {
            System.gc();
            InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, "OOM Exception trying to remove large logs...", e);
            e.printStackTrace();
            try {
                // try to remove logs
                bodyJsonObject.remove("console_log");
                bodyJsonObject.remove("instabug_log");
                bodyJsonObject.remove("network_log");
                return bodyJsonObject.toString();
            } catch (OutOfMemoryError totalOOMException) {
                InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, "Failed to resolve OOM, returning empty request body", e);
                totalOOMException.printStackTrace();
            }
        }
        return "{}";
    }

    @Nullable
    public FileToUpload getFileToUpload() {
        return fileToUpload;
    }

    @Nullable
    public File getDownloadedFile() {
        return downloadedFile;
    }


    /**
     * Get a builder instance to modify the request
     */
    public Builder builder() {
        return new Builder()
                .endpoint(this.endPoint)
                .url(this.requestUrl)
                .method(this.requestMethod)
                .type(this.requestType)
                .shorten(shorten)
                .fileToUpload(this.fileToUpload)
                .fileToDownload(this.downloadedFile)
                .setBodyParameter(this.bodyParameters)
                .setUrlParameter(this.urlParameters)
                .setHeaders(this.headers);
    }

    @NonNull
    @Override
    public String toString() {
        if (requestMethod != null && requestMethod.equals(RequestMethod.GET)) {
            return "Url: " + getRequestUrl() + " | Method: " + requestMethod;
        } else {
            return "Url: " + getRequestUrl() + " | Method: " + requestMethod + " | Body: " + getRequestBody();
        }
    }

    public boolean isMultiPartRequest() {
        return fileToUpload != null;
    }

    public interface Callbacks<T, K> {
        void onSucceeded(T response);

        void onFailed(K error);

        default void onDisconnected() {
        }

        default void onRetrying(Throwable throwable) {
        }
    }

    public static class Builder {

        @Nullable
        private String requestUrl;
        @Nullable
        private String endpoint;
        @Nullable
        @RequestMethod
        private String method;
        @RequestType
        private int type = RequestType.UNDEFINED;
        @Nullable
        private ArrayList<RequestParameter> urlParameters;
        @Nullable
        private ArrayList<RequestParameter> bodyParameters;
        @Nullable
        private ArrayList<RequestParameter<String>> headers;
        @Nullable
        private FileToUpload fileToUpload;
        @Nullable
        private File downloadedFile;
        @Nullable
        private JSONObject requestBodyJsonObject;
        private boolean shorten;
        private boolean hasUuid = true;
        private boolean noDefaultParameters = false;

        private AppTokenProvider tokenProvider = new AppTokenProvider() {
            @Nullable
            @Override
            public String getAppToken() {
                return TokenMappingServiceLocator.getTokenMappingConfigs().getAvailableAppToken();
            }
        };

        public Builder() {
            addHeader(new RequestParameter<>(SYSTEM_PLATFORM_OS, "android"));
            addHeader(new RequestParameter<>(ANDROID_VERSION, Build.VERSION.RELEASE));
            addHeader(new RequestParameter<>(SDK_VERSION, DeviceStateProvider.getSdkVersion()));
            String appVariant = SettingsManager.getInstance().getAppVariant();
            if (appVariant != null) {
                addHeader(new RequestParameter<>(APP_VARIANT, appVariant));
            }
        }

        public Builder tokenProvider(AppTokenProvider tokenProvider) {
            this.tokenProvider = tokenProvider;
            return this;
        }

        /**
         * Set the request url if it isn't based of the SDK base url
         *
         * @return Builder instance
         * @see Constants#BASE_URL
         */
        public Builder url(String url) {
            this.requestUrl = url;
            return this;
        }

        /**
         * Set the request endpoint to be attached to the SDK base url
         *
         * @return Builder instance
         * @see Constants#BASE_URL
         */
        public Builder endpoint(@Nullable String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        /**
         * Set the request method {@link RequestMethod}
         *
         * @return Builder instance
         */
        public Builder method(@Nullable @RequestMethod String requestMethod) {
            this.method = requestMethod;
            return this;
        }

        /**
         * Set the request type {@link RequestType}
         *
         * @return Builder instance
         */
        public Builder type(@RequestType int requestType) {
            this.type = requestType;
            return this;
        }

        /**
         * Add url parameter {@link RequestParameter} to the request if the {@link RequestMethod} is GET or DELETE, otherwise,
         * add body parameter
         *
         * @return Builder instance
         */
        public Builder addParameter(RequestParameter requestParameter) {
            if (method != null) {
                if (method.equals(RequestMethod.GET) || method.equals(RequestMethod.DELETE)) {
                    addUrlParameter(requestParameter);
                } else {
                    addBodyParameter(requestParameter);
                }
            }
            return this;
        }

        public Builder setParameter(List<RequestParameter> requestParameters) {
            if (method != null) {
                if (method.equals(RequestMethod.GET) || method.equals(RequestMethod.DELETE)) {
                    setUrlParameter(requestParameters);
                } else {
                    setBodyParameter(requestParameters);
                }
            }
            return this;
        }

        /**
         * Add url parameter {@link RequestParameter} to the request if the {@link RequestMethod} is GET or DELETE, otherwise,
         * add body parameter
         *
         * @return Builder instance
         */
        private Builder addUrlParameter(RequestParameter parameter) {

            if (urlParameters == null) {
                urlParameters = new ArrayList<>();
            }
            urlParameters.add(parameter);
            return this;
        }

        /**
         * Sets url parameters {@link RequestParameter} to the request
         *
         * @param parameters url parameters list
         * @return Builder instance
         */
        private Builder setUrlParameter(List<RequestParameter> parameters) {

            if (urlParameters == null) {
                urlParameters = new ArrayList<>();
            }
            urlParameters = new ArrayList<>(parameters);
            return this;
        }

        /**
         * Add body parameter {@link RequestParameter} to the request
         *
         * @return Builder instance
         */
        private Builder addBodyParameter(RequestParameter parameter) {
            if (bodyParameters == null) {
                bodyParameters = new ArrayList<>();
            }
            bodyParameters.add(parameter);
            return this;
        }

        /**
         * Sets body parameters {@link RequestParameter} to the request
         *
         * @param parameters body parameters list
         * @return Builder instance
         */
        private Builder setBodyParameter(List<RequestParameter> parameters) {
            if (bodyParameters == null) {
                bodyParameters = new ArrayList<>();
            }
            bodyParameters = new ArrayList<>(parameters);
            return this;
        }

        /**
         * Specify if the base params keys are shorted or sen in full name
         *
         * @return Builder instance
         * @see Request#handleBaseParams
         */
        public Builder shorten(boolean isShortedKeys) {
            this.shorten = isShortedKeys;
            return this;
        }

        public Builder hasUuid(boolean hasUuid) {
            this.hasUuid = hasUuid;
            return this;
        }

        /**
         * Add header {@link RequestParameter} to the request
         *
         * @return Builder instance
         */
        public Builder addHeader(RequestParameter<String> header) {
            if (headers == null) {
                headers = new ArrayList<>();
            }
            headers.add(header);
            return this;
        }


        /**
         * Add header {@link RequestParameter} to the request
         *
         * @return Builder instance
         */
        public Builder setHeaders(List<RequestParameter<String>> headers) {
            if (headers == null) {
                headers = new ArrayList<>();
            }
            this.headers = new ArrayList<>(headers);
            return this;
        }

        /**
         * Set the file to be uploaded in the request
         *
         * @return Builder instance
         * @see FileToUpload
         */
        public Builder fileToUpload(@Nullable FileToUpload fileToUpload) {
            this.fileToUpload = fileToUpload;
            return this;
        }

        /**
         * Set the path of the downloading file
         *
         * @return Builder instance
         */
        public Builder fileToDownload(@Nullable File file) {
            this.downloadedFile = file;
            return this;
        }

        /**
         * Specify if the base params should be added to request or not
         *
         * @return Builder instance
         */
        public Builder disableDefaultParameters(boolean noDefaultParameters) {
            this.noDefaultParameters = noDefaultParameters;
            return this;
        }

        /**
         * Set a JsonObject as a base for the Request parameters
         * @return Builder instance
         */
        public Builder setRequestBodyJsonObject(@Nullable JSONObject requestBodyJsonObject) {
            this.requestBodyJsonObject = requestBodyJsonObject;
            return this;
        }

        /**
         * Build the request with the provided info without adding UUID
         *
         * @return new instance of the {@link Request}
         */
        public Request build() {
            addTokenToHeaders();
            return new Request(this);
        }

        private void addTokenToHeaders() {
            String token = tokenProvider.getAppToken();
            if (token != null) {
                addHeader(new RequestParameter<String>(Header.APP_TOKEN, token));
            }
        }

    }
}