package com.instabug.library.networkv2.limitation

import com.instabug.library.internal.sharedpreferences.corePref
import java.util.concurrent.TimeUnit

/**
 * A contract to be implemented by settings provider. Instance of this class is to be used
 * in [RateLimiter]
 */
interface RateLimitationSettings {
    val isRateLimited: Boolean
    fun setLastRequestStartedAt(time: Long)
    fun setLimitedUntil(period: Int)
}

class RateLimitationSettingsImpl(val feature: RateLimitedFeature) :
    RateLimitationSettings {
    private var _limitedUntil: Long by corePref(
        key = feature.featureName + LIMITED_UNTIL_KEY,
        defaultValue = 0
    )

    private var _lastRequestStartedAt: Long by corePref(
        key = feature.featureName + REQUEST_STARTED_AT_KEY,
        defaultValue = 0
    )

    override fun setLimitedUntil(period: Int) {
        _limitedUntil = _lastRequestStartedAt + TimeUnit.SECONDS.toMillis(period.toLong())
    }

    override fun setLastRequestStartedAt(time: Long) {
        _lastRequestStartedAt = time
    }

    override val isRateLimited: Boolean
        get() {
            val start = _lastRequestStartedAt
            val limitedUntil = _limitedUntil
            val currentTime = System.currentTimeMillis()
            return start != 0L && limitedUntil != 0L && currentTime > start && currentTime < limitedUntil
        }

    companion object {
        const val LIMITED_UNTIL_KEY = "_limited_until"
        const val REQUEST_STARTED_AT_KEY = "_request_started_at"
    }
}

enum class RateLimitedFeature(val featureName: String) {
    V3_SESSION("v3_sessions"),
    SESSION_REPLAY("session-replay")
}