package com.instabug.library.model.v3Session

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.Instabug
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_APP_TOKEN
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_APP_VERSION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_BACKGROUND_START_TIME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_DEVICE
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_DURATION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_FOREGROUND_START_TIME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_ID
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_LOCALE
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_NANO_START_TIME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_OS
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_PRODUCTION_USAGE
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_RATING_DIALOG_DETECTION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SCREEN_SIZE
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SDK_VERSION
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SERIAL
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SESSION_RANDOM_ID
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SESSION_STITCHING_STATE
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SR_ENABLED
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SR_EVALUATED
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_SYNC_STATUS
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_USERS_PAGE_ENABLED
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_USER_ATTRIBUTES
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_USER_EMAIL
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_USER_EVENTS
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_USER_NAME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_UUID
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionEntry.COLUMN_V2_SESSION_SENT
import com.instabug.library.model.common.Session
import com.instabug.library.model.common.SessionVersion
import com.instabug.library.model.v3Session.AppDataKeys.BUNDLE_ID_KEY
import com.instabug.library.model.v3Session.AppDataKeys.DEBUG_MODE_KEY
import com.instabug.library.model.v3Session.AppDataKeys.DENSITY_KEY
import com.instabug.library.model.v3Session.Defaults.DEFAULT_STORE_URL
import com.instabug.library.model.v3Session.ProductionUsageKeys.APM_PRODUCTION_USAGE_KEY
import com.instabug.library.model.v3Session.ProductionUsageKeys.BUGS_PRODUCTION_USAGE_KEY
import com.instabug.library.model.v3Session.ProductionUsageKeys.CRASHES_PRODUCTION_USAGE_KEY
import com.instabug.library.model.v3Session.ProductionUsageKeys.FEATURES_REQUESTS_PRODUCTION_USAGE_KEY
import com.instabug.library.model.v3Session.ProductionUsageKeys.STORE_URL_KEY
import com.instabug.library.model.v3Session.ProductionUsageKeys.SURVEYS_PRODUCTION_USAGE_KEY
import com.instabug.library.model.v3Session.SessionsRequestKeys.DROPPED_SESSIONS_COUNT_KEY
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.Header
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestMethod
import com.instabug.library.networkv2.request.RequestParameter
import com.instabug.library.sessionV3.configurations.IBGSessionConfigurations
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.util.DeviceStateProvider
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.extenstions.asInt
import com.instabug.library.util.extenstions.getBoolean
import com.instabug.library.util.extenstions.getLong
import com.instabug.library.util.extenstions.getNullableString
import com.instabug.library.util.extenstions.getString
import org.json.JSONArray
import org.json.JSONObject
import java.util.concurrent.TimeUnit


object IBGSessionMapper {
    fun IBGSession.toCoreSession(): Session = object : Session {
        override fun getId(): String = this@toCoreSession.id

        override fun getOs(): String = appData.os

        override fun getUuid(): String = userData.uuid

        override fun getAppVersion(): String? = appData.appVersion

        override fun getStartTimestampMicros(): Long = startTime.value

        override fun getStartNanoTime(): Long = startTime.startNanoTime

        @SessionVersion
        override fun getVersion(): String = SessionVersion.V3
    }

    fun IBGSession.toContentValues(): IBGContentValues = IBGContentValues().apply {
        put(COLUMN_ID, id, true)
        put(COLUMN_DURATION, durationInMicro, false)
        put(COLUMN_V2_SESSION_SENT, isV2SessionSent.asInt, false)
        put(COLUMN_SESSION_STITCHING_STATE, stitchingState.name, false)
        put(COLUMN_SYNC_STATUS, syncStatus.name, true)
        put(COLUMN_PRODUCTION_USAGE, productionUsage?.asJson, false)
        put(COLUMN_SESSION_RANDOM_ID, randomID.toLong(), true)
        put(COLUMN_SR_ENABLED, srEnabled.asInt, true)
        put(COLUMN_SR_EVALUATED, isSrEvaluated.asInt, true)
        addStartTimeValues(startTime)
        addUserDataValues(userData)
        addAppDataValues(appData)
        addStoreRatingData(ratingDialogDetection)
    }

    private fun IBGContentValues.addAppDataValues(appData: SessionAppData) = with(appData) {
        put(COLUMN_APP_TOKEN, appToken, false)
        put(COLUMN_OS, os, false)
        put(COLUMN_DEVICE, device, false)
        put(COLUMN_SDK_VERSION, sdkVersion, false)
        put(COLUMN_APP_VERSION, appVersion, false)
        put(COLUMN_LOCALE, locale, false)
        put(COLUMN_SCREEN_SIZE, screenSize, false)
    }

    private fun IBGContentValues.addUserDataValues(
        userData: SessionUserData
    ) = with(userData) {
        put(COLUMN_UUID, uuid, true)
        put(COLUMN_USER_EVENTS, userEvents, false)
        put(COLUMN_USER_ATTRIBUTES, customAttributes, false)
        put(COLUMN_USER_EMAIL, userEmail, false)
        put(COLUMN_USER_NAME, userName ?: "", false)
        put(COLUMN_USERS_PAGE_ENABLED, usersPageEnabled.asInt, false)

    }

    private fun IBGContentValues.addStartTimeValues(
        startTime: StartTime
    ) = with(startTime) {
        put(COLUMN_BACKGROUND_START_TIME, backgroundMicroStartTime, false)
        put(COLUMN_NANO_START_TIME, startNanoTime, false)
        put(COLUMN_FOREGROUND_START_TIME, foregroundMicroStartTime, false)
    }

    private fun IBGContentValues.addStoreRatingData(ratingDialogDetection: String?) {
        ratingDialogDetection?.let {
            put(COLUMN_RATING_DIALOG_DETECTION, it, false)
        }
    }

    fun IBGCursor.toSession() = use {
        takeIf { moveToNext() }?.session
    }

    fun IBGCursor.toSessionList(): IBGSessions = use {
        mutableListOf<IBGSession>().apply {
            while (moveToNext()) {
                add(session)
            }
        }
    }

    private val IBGCursor.session: IBGSession
        get() = IBGSession(
            serial = getLong(COLUMN_SERIAL),
            id = getString(COLUMN_ID),
            userData = sessionUserData,
            appData = appData,
            stitchingState = StitchingState.valueOf(getString(COLUMN_SESSION_STITCHING_STATE)),
            durationInMicro = getLong(COLUMN_DURATION),
            productionUsage = getNullableString(COLUMN_PRODUCTION_USAGE)?.toSessionProductionUsage,
            startTime = statTime,
            isV2SessionSent = getBoolean(COLUMN_V2_SESSION_SENT),
            syncStatus = SyncStatus.valueOf(getString(COLUMN_SYNC_STATUS)),
            randomID = getLong(COLUMN_SESSION_RANDOM_ID).toUInt(),
            srEnabled = getBoolean(COLUMN_SR_ENABLED),
            isSrEvaluated = getBoolean(COLUMN_SR_EVALUATED),
            ratingDialogDetection = getNullableString(COLUMN_RATING_DIALOG_DETECTION)
        )

    private val IBGCursor.sessionUserData
        get() = SessionUserData(
            uuid = getString(COLUMN_UUID),
            userName = getNullableString(COLUMN_USER_NAME),
            userEmail = getNullableString(COLUMN_USER_EMAIL),
            usersPageEnabled = getBoolean(COLUMN_USERS_PAGE_ENABLED),
            userEvents = getNullableString(COLUMN_USER_EVENTS),
            customAttributes = getNullableString(COLUMN_USER_ATTRIBUTES),
        )

    private val IBGCursor.statTime
        get() = StartTime(
            startNanoTime = getLong(COLUMN_NANO_START_TIME),
            foregroundMicroStartTime = getLong(COLUMN_FOREGROUND_START_TIME),
            backgroundMicroStartTime = getLong(COLUMN_BACKGROUND_START_TIME),
        )

    private val IBGCursor.appData
        get() = SessionAppData(
            device = getString(COLUMN_DEVICE),
            sdkVersion = getNullableString(COLUMN_SDK_VERSION),
            appToken = getNullableString(COLUMN_APP_TOKEN),
            appVersion = getNullableString(COLUMN_APP_VERSION),
            os = getString(COLUMN_OS),
            locale = getNullableString(COLUMN_LOCALE),
            screenSize = getNullableString(COLUMN_SCREEN_SIZE)
        )

    @VisibleForTesting
    val SessionProductionUsage.asJson
        get() = hashMapOf<String, Any>()
            .let(::extractFields)
            .let(::JSONObject)
            .toString()

    @VisibleForTesting
    val String.toSessionProductionUsage
        get() = JSONObject(this).run {
            SessionProductionUsage(
                storeURL = optString(STORE_URL_KEY, DEFAULT_STORE_URL),
                apm = optBoolean(APM_PRODUCTION_USAGE_KEY, false),
                bugs = optBoolean(BUGS_PRODUCTION_USAGE_KEY, false),
                crashes = optBoolean(CRASHES_PRODUCTION_USAGE_KEY, false),
                featureRequest = optBoolean(FEATURES_REQUESTS_PRODUCTION_USAGE_KEY, false),
                surveys = optBoolean(SURVEYS_PRODUCTION_USAGE_KEY, false)
            )

        }

    fun IBGSessionDTO.constructRequest(configurations: IBGSessionConfigurations = IBGSessionServiceLocator.sessionConfigurations): Request? =
        Request.Builder().url(Endpoints.V3_SESSION)
            .method(RequestMethod.POST)
            .addDroppedSessionCount(configurations)
            .addCommonKeys(commonKeys)
            .shorten(true)
            .addStaticData()
            .addSessions(sessions)
            .addSessionReplayCountHeader(sessionReplayCount)
            .addDebugMode(configurations)
            .disableDefaultParameters(true)
            .build()

    private fun Request.Builder.addCommonKeys(common: IBGSessionMap) = apply {
        common
            .forEach { entry ->
                addParameter(RequestParameter(entry.key, mapValueToJson(entry.value)))
            }
    }

    private fun Request.Builder.addSessionReplayCountHeader(sessionReplayCount: Int) = apply {
        addHeader(
            RequestParameter(
                Header.SESSION_REPLAY_COUNT,
                sessionReplayCount.toString()
            )
        )
    }

    private fun Request.Builder.addSessions(sessions: List<IBGSessionMap>) = apply {
        sessions
            .map(::mapSessionToJson)
            .let { sessionsList ->
                addParameter(
                    RequestParameter(SessionsRequestKeys.SESSIONS_KEY, JSONArray(sessionsList))
                )
            }
    }

    private fun mapSessionToJson(sessionMap: IBGSessionMap) = sessionMap
        .map { sessionProp -> sessionProp.key to mapValueToJson(sessionProp.value) }
        .toMap()

    private fun mapValueToJson(value: Any) =
        if (isJsonArray(value)) JSONArray(value.toString())
        else if (isJsonObject(value)) JSONObject(value.toString())
        else value

    private fun isJsonArray(value: Any) =
        value.toString().startsWith("[") && value.toString().endsWith("]")

    private fun isJsonObject(value: Any) =
        value.toString().startsWith("{") && value.toString().endsWith("}")

    private fun Request.Builder.addDroppedSessionCount(ibgSessionConfigurations: IBGSessionConfigurations) =
        apply {
            val droppedSessionCount = ibgSessionConfigurations.droppedSessionCount
            if (droppedSessionCount > 0) {
                InstabugSDKLogger.w(
                    LOG_TAG,
                    "$droppedSessionCount sessions have been dropped due to reaching sessions storage limit. Please contact support for more information."
                )
                addParameter(
                    RequestParameter(
                        DROPPED_SESSIONS_COUNT_KEY,
                        droppedSessionCount
                    )
                )
            }


        }

    private fun Request.Builder.addDebugMode(ibgSessionConfigurations: IBGSessionConfigurations) =
        apply {
            if (ibgSessionConfigurations.inDebugMode) {
                addHeader(RequestParameter(Header.DEBUG_MODE_HEADER, "true"))
                addParameter(RequestParameter(DEBUG_MODE_KEY, true))
            }
        }

    private fun Request.Builder.addStaticData() =
        apply {
            val applicationContext = Instabug.getApplicationContext()
            val density = DeviceStateProvider.getScreenDensity(applicationContext)
            addParameter(RequestParameter(DENSITY_KEY, density))
            val bundleID = DeviceStateProvider.getAppPackageName(applicationContext)
            addParameter(RequestParameter(BUNDLE_ID_KEY, bundleID))

        }

    val IBGSessionData.asPair
        get() = featureKey to featureData

    val IBGInMemorySession.asForegroundStartEvent: IBGSdkCoreEvent.V3Session.V3StartedInForeground
        get() = IBGSdkCoreEvent.V3Session.V3StartedInForeground(
            startTime = startTime.foregroundMicroStartTime.let(TimeUnit.MICROSECONDS::toMillis),
            uuid = id,
            partialId = randomID
        )

    fun IBGInMemorySession.getCompositeSessionId(appToken: String): String? =
        startTime.takeUnless(StartTime::isBackground)
            ?.foregroundMicroStartTime
            ?.let(TimeUnit.MICROSECONDS::toMillis)
            ?.let { fgStart -> "$appToken-$fgStart-$randomID" }
}
