package com.instabug.library.logging.disklogs;

import android.annotation.SuppressLint;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Instabug;
import com.instabug.library.InstabugState;
import com.instabug.library.InstabugStateProvider;
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent;
import com.instabug.library.internal.resolver.LoggingSettingResolver;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.model.LogDescriptor;
import com.instabug.library.model.LoggingSettings;
import com.instabug.library.model.LoggingSettings.LogLevels;

import java.util.concurrent.Executor;

/**
 * A middle ware logger to control the logs based on its settings.
 */
public final class InstabugSDKDiskLogger {

    @Nullable
    private IBGLoggingThread ibgLoggingThread;
    private LoggingSettingResolver loggingSettingResolver;
    private final FileDiskOperator logFileDiskOperator;
    private Executor loggingExecutor = CoreServiceLocator.getLogsFileExecutor();

    public InstabugSDKDiskLogger() {
        ibgLoggingThread = new IBGLoggingThread();
        loggingSettingResolver = LoggingSettingResolver.getInstance();
        logFileDiskOperator = CoreServiceLocator.getFileDiskOperator();
    }


    /**
     * Log base logs as long as log's level != {@link LogLevels#NO_LOGS}
     * in this level it will log
     * 1. debug
     * 2. warning
     * 3. errors
     * 4. info
     *
     * @param tag
     * @param message
     * @param currentThreadName
     */
    @SuppressLint("THREAD_SAFETY_VIOLATION")
    public void logBaseLevel(String tag, String message, String currentThreadName, long timestamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS
                            && ibgLoggingThread != null) {
                        ibgLoggingThread.addLog(tag, message, currentThreadName, timestamp);
                    }
                }
            }
        });
    }


    /**
     * Log base logs as long as log's level == {@link LogLevels#LEVEL_TWO}
     * in this level it will log
     * 1. debug
     * 2. warning
     * 3. errors
     * 4. info
     * 5. verbose
     *
     * @param tag
     * @param message
     * @param currentThreadName
     */
    public void logVerboseLogs(String tag, String message, String currentThreadName, long timestamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() == LogLevels.LEVEL_TWO
                            && ibgLoggingThread != null) {
                        ibgLoggingThread.addLog(tag, message, currentThreadName, timestamp);
                    }
                }
            }
        });
    }


    public void logSessionDetails(LogDescriptor logDescriptor) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS) {
                        if (ibgLoggingThread != null)
                            ibgLoggingThread.logSessionDescriptor(logDescriptor);
                    }
                }
            }
        });
    }

    @VisibleForTesting
    void setLoggingSettingResolver(LoggingSettingResolver loggingSettingResolver) {
        this.loggingSettingResolver = loggingSettingResolver;
    }

    @VisibleForTesting
    void setibgLoggingThread(IBGLoggingThread ibgLoggingThread) {
        this.ibgLoggingThread = ibgLoggingThread;
    }

    public void logEndSession(long timeStamp) {
        loggingExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (InstabugStateProvider.getInstance().getState() != InstabugState.DISABLED) {
                    LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
                    if (loggingSettings != null && loggingSettings.getLevel() != LogLevels.NO_LOGS) {
                        if (ibgLoggingThread != null)
                            ibgLoggingThread.logEndSession(timeStamp);
                    }
                }
            }
        });
    }

    public void onDiskLoggingSettingsChanged(LoggingSettings loggingSettings) {
        if (loggingSettings.getLevel() == LoggingSettings.LogLevels.NO_LOGS) {
            if (ibgLoggingThread != null) {
                ibgLoggingThread.interrupt();
                ibgLoggingThread = null;
            }
            logFileDiskOperator.clear();
        } else {
            if (ibgLoggingThread == null && Instabug.getApplicationContext() != null) {
                ibgLoggingThread = new IBGLoggingThread();
            }
        }
        if (loggingSettings.isShouldClearDiskLogs()) {
            logFileDiskOperator.clear();
        }
    }


    @VisibleForTesting
    void setLoggingExecutor(Executor loggingExecutor) {
        this.loggingExecutor = loggingExecutor;
    }

    public void handleCoreEvents(IBGSdkCoreEvent coreEvent) {
        if (coreEvent == IBGSdkCoreEvent.SdkVersionChanged.INSTANCE) {
            LoggingSettings loggingSettings = loggingSettingResolver.getLoggingSettings();
            if (loggingSettings != null) {
                loggingSettings.setShouldClearDiskLogs(true);
            }
            logFileDiskOperator.clear();
        }
    }
}
