package com.instabug.library.internal.sharedpreferences

import android.content.SharedPreferences
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty

//TODO: Consider supporting nullable types (i.e. String?)
@Suppress("UNCHECKED_CAST")
abstract class PreferencesProperty<T>(
    protected val key: String,
    protected val defaultValue: T
) : ReadWriteProperty<Any?, T> {
    @Volatile
    private var value = defaultValue

    @Volatile
    private var firstLoad: Boolean = true
    abstract val pref: SharedPreferences?
    override fun getValue(thisRef: Any?, property: KProperty<*>): T = synchronized(this) {
        takeIf { firstLoad }
            ?.also { firstLoad = false }
            ?.let { pref?.get() ?: value }
            ?.also { value = it }
            ?: value
    }

    //TODO: Need to revise nullability support for this case (i.e. passed defaultValue is null or of nullable type).
    protected open fun SharedPreferences.get() =
        when (defaultValue) {
            is String -> getString(key, defaultValue)
            is Float -> getFloat(key, defaultValue)
            is Int -> getInt(key, defaultValue)
            is Long -> getLong(key, defaultValue)
            is Boolean -> getBoolean(key, defaultValue)
            is Set<*> -> getStringSet(key, defaultValue as Set<String>)
            else -> throw UnsupportedOperationException("can't persist non-primitive type")
        } as T

    override fun setValue(thisRef: Any?, property: KProperty<*>, value: T) {
        synchronized(this) {
            firstLoad = false
            this.value = value
        }
        persistValue(value)
    }

    protected fun persistValue(value: T) {
        pref?.edit()?.put(value)?.apply()
    }

    open fun clear() {
        value = defaultValue
        clearPersistedValue()
    }

    protected fun clearPersistedValue() {
        pref?.edit()?.remove(key)?.apply()
    }

    //TODO: Need to revise nullability support for this case (i.e. passed value is null or of nullable type).
    protected open fun SharedPreferences.Editor.put(value: T): SharedPreferences.Editor {
        when (value) {
            is String -> putString(key, value)
            is Float -> putFloat(key, value)
            is Int -> putInt(key, value)
            is Long -> putLong(key, value)
            is Boolean -> putBoolean(key, value)
            is Set<*> -> putStringSet(key, value as Set<String>)
        }
        return this
    }
}

interface PreferencePropertyFactory {
    fun <T> create(key: String, defaultValue: T): PreferencesProperty<T>
}
