package com.instabug.library.internal.filestore

/**
 * An operation builder scope giving the ability to intuitively scope an operation
 * in directories and on spans.
 */
class OperationScopeBuilder<OpIn : Directory, OpOut>(
    private val operation: FileOperation<OpIn, OpOut>
) {
    /**
     * Scopes the given [operation] given that the execution input
     * is the parent directory of the basic operation input operations directory.
     * @param directorySelector a [DirectorySelector] with input matching the final execution input-type
     * and output-type matching the basic operation input-type (conversion).
     */
    fun <SIn : Directory> inDirectory(
        directorySelector: DirectorySelector<SIn, OpIn>
    ) = OperationScopeBuilder(OperateOnDirectory(directorySelector, operation))

    /**
     * Chains multiple operations in the order of building mutating the output expected to the new op.
     * NOTE: All operations submitted back-to-back using this scope builder should be operating on the same
     * input directory, otherwise, [inDirectory] should be used to separate between them for directory traversal.
     * @param newOperation The new [FileOperation] to chain on the back of the current operation.
     */
    fun <NOpOut> continueWith(newOperation: FileOperation<OpIn, NOpOut>) =
        OperationScopeBuilder(ContinueWithOp(operation, newOperation))

    /**
     * Scopes the given [operation] to a span given that the execution input is a [SpansDirectory].
     *@param spanSelector a [SpanSelector] with input matching the final execution input-type
     * extending [SpansDirectory] and a [SpansDir] matching output-type (selection).
     */
    fun <SpansDir : SpansDirectory<OpIn>> onSpan(
        spanSelector: SpanSelector<SpansDir, OpIn>
    ) = OperationSpanScopeBuilder(spanSelector, operation)

    /**
     * Scopes the given [operation] to a number of spans given that the execution input is a [SpansDirectory].
     *@param spanSelector a [SpanSelector] with input matching the final execution input-type
     * extending [SpansDirectory] and a list of [SpansDir] matching output-type (selection).
     */
    fun <SpansDir : SpansDirectory<OpIn>> onMultiSpans(
        spanSelector: MultiSpanSelector<SpansDir, OpIn>
    ) = OperationMultiSpanScopeBuilder(spanSelector, operation)

    fun build(): FileOperation<OpIn, OpOut> = operation

    fun buildAndExecute(input: OpIn?): OpOut? = input?.let { build().invoke(it) }
}

class OperationSpanScopeBuilder<SpansDir : SpansDirectory<OpIn>, OpIn : Directory, OpOut>(
    private val spanSelector: SpanSelector<SpansDir, OpIn>,
    private val operation: FileOperation<OpIn, OpOut>
) {
    fun build() = OperateOnSpan(spanSelector, operation)

    fun buildAndExecute(input: SpansDir?): OpOut? = input?.let { build().invoke(it) }
}

class OperationMultiSpanScopeBuilder<SpansDir : SpansDirectory<OpIn>, OpIn : Directory, OpOut>(
    private val spansSelector: MultiSpanSelector<SpansDir, OpIn>,
    private val operation: FileOperation<OpIn, OpOut>
) {
    fun build() = OperateOnMultiSpans(spansSelector, operation)

    fun buildAndExecute(input: SpansDir?): List<OpOut>? = input?.let { build().invoke(it) }
}