package com.instabug.library.diagnostics.nonfatals.networking.mapping;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.diagnostics.nonfatals.model.Occurrence;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.user.UserManager;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

/**
 * Created by Shannan on 06/09/2021.
 */
public final class NonFatalMapperImpl implements NonFatalMapper {

    private static final String INSTABUG_PREFIX = "com.instabug.";

    @NonNull
    @Override
    public JSONArray toJSONArray(@NonNull List<NonFatal> nonFatals) throws JSONException {
        JSONArray jsonArray = new JSONArray();
        for (NonFatal nonFatal : nonFatals) {
            JSONObject nonFatalJson = new JSONObject();
            if (nonFatal != null && nonFatal.getState() != null) {
                nonFatalJson.put(MappingKeys.SDK_VERSION, nonFatal.getState().getSdkVersion());
                nonFatalJson.put(MappingKeys.BUNDLE_ID, nonFatal.getState().getAppPackageName());
                nonFatalJson.put(MappingKeys.CARRIER, nonFatal.getState().getCarrier());
                nonFatalJson.put(MappingKeys.DENSITY, nonFatal.getState().getScreenDensity());
                nonFatalJson.put(MappingKeys.DEVICE, nonFatal.getState().getDevice());
                nonFatalJson.put(MappingKeys.SCREEN_SIZE, nonFatal.getState().getScreenSize());
                nonFatalJson.put(MappingKeys.LOCALE, nonFatal.getState().getLocale());
                nonFatalJson.put(MappingKeys.OS, nonFatal.getState().getOS());
                if (nonFatal.getState().getDevicePerformanceClass() > -1)
                    nonFatalJson.put(MappingKeys.DEVICE_PERFORMANCE_CLASS, nonFatal.getState().getDevicePerformanceClass());

                JSONObject userAttributesJson = new JSONObject();
                userAttributesJson.put(MappingKeys.MODULE_AFFECTED, getAffectedModule(nonFatal.getDeclaringClass()));
                userAttributesJson.put(MappingKeys.APP_VERSION, nonFatal.getState().getAppVersion());
                userAttributesJson.put(MappingKeys.USER_UUID, UserManager.getUUID());
                userAttributesJson.put(MappingKeys.APPLICATION_TOKEN, SettingsManager.getInstance().getAppToken());
                nonFatalJson.put(MappingKeys.USER_ATTRIBUTES, userAttributesJson);


                JSONObject errorDetailsJson = new JSONObject();
                JSONObject errorJson = new JSONObject();

                errorJson.put(MappingKeys.NAME, nonFatal.getExceptionType());
                errorJson.put(MappingKeys.EXCEPTION, nonFatal.getExceptionType() + ":" + nonFatal.getMessage());
                errorJson.put(MappingKeys.MESSAGE, nonFatal.getMessage());
                errorJson.put(MappingKeys.LOCATION, nonFatal.getDeclaringClass() + ":" + nonFatal.getLineNumber());
                errorJson.put(MappingKeys.STACKTRACE, nonFatal.getStackTrace());

                errorDetailsJson.put(MappingKeys.ERROR, errorJson);
                nonFatalJson.put(MappingKeys.ERROR_DETAILS, errorDetailsJson);

                nonFatalJson.put(MappingKeys.UUID, SettingsManager.getInstance().getUuid());

                JSONArray occurrencesJsonArray = new JSONArray();


                for (int i = 0; i < nonFatal.getOccurrences().size(); i++) {
                    JSONObject occurrenceJson = new JSONObject();
                    Occurrence occurrence = nonFatal.getOccurrences().get(i);
                    occurrenceJson.put(MappingKeys.REPORTED_AT, occurrence.getReportedAt());

                    if (occurrence.getState() != null) {
                        occurrenceJson.put(MappingKeys.WIFI_STATE, occurrence.getState().isWifiEnable());
                        occurrenceJson.put(MappingKeys.APP_STATUS, occurrence.getState().getAppStatus());
                        occurrenceJson.put(MappingKeys.BATTERY_LEVEL, occurrence.getState().getBatteryLevel());
                        occurrenceJson.put(MappingKeys.BATTERY_STATE, occurrence.getState().getBatteryState());
                        occurrenceJson.put(MappingKeys.CURRENT_VIEW, occurrence.getState().getCurrentView());
                        occurrenceJson.put(MappingKeys.DURATION, occurrence.getState().getDuration());
                        occurrenceJson.put(MappingKeys.MEMORY_TOTAL, occurrence.getState().getTotalMemory());
                        occurrenceJson.put(MappingKeys.MEMORY_USED, occurrence.getState().getUsedMemory());
                        occurrenceJson.put(MappingKeys.ORIENTATION, occurrence.getState().getScreenOrientation());
                        occurrenceJson.put(MappingKeys.STORAGE_TOTAL, occurrence.getState().getTotalStorage());
                        occurrenceJson.put(MappingKeys.STORAGE_USED, occurrence.getState().getUsedStorage());
                    }

                    occurrencesJsonArray.put(occurrenceJson);
                }

                nonFatalJson.put(MappingKeys.OCCURRENCES, occurrencesJsonArray);
            }
            if (nonFatalJson.length() != 0) {
                jsonArray.put(nonFatalJson);
            }
        }
        return jsonArray;
    }

    @VisibleForTesting
    String getAffectedModule(@Nullable String declaringClass) {
        if (declaringClass != null && !declaringClass.isEmpty() && declaringClass.startsWith(INSTABUG_PREFIX)) {
            String trimmedClassName = declaringClass.substring(INSTABUG_PREFIX.length());
            return trimmedClassName.substring(0, trimmedClassName.indexOf('.'));
        }
        return "";
    }
}
