package com.instabug.library.diagnostics.customtraces

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.Instabug
import com.instabug.library.diagnostics.DiagnosticsConstants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.diagnostics.customtraces.utils.CustomTracesValidator
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import kotlin.math.min

object IBGPendingTraceHandler {

    @JvmStatic
    var builderFGStartTime: Long = 0

    @JvmStatic
    var builderFGEndTime: Long = 0

    @JvmStatic
    var builderBGStartTime: Long = 0

    @JvmStatic
    var builderConstructorStartTime: Long = 0

    @JvmStatic
    var builderConstructorEndTime: Long = 0

    @JvmStatic
    var builderBGEndTime: Long = 0
        set(value) {
            synchronized(this) {
                field = value
                if (Instabug.isEnabled()) {
                    if (CustomTracesValidator.shouldSaveTrace(DiagnosticsConstants.BUILDER_BG)) {
                        builderBGStartTime *= 1000
                        field *= 1000
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.BUILDER_BG, builderBGStartTime, field
                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.BUILDER_BG} trace executed in ${field - builderBGStartTime} microseconds"
                        )
                        field = 0L
                        flushTraces()
                    }
                }
            }
        }

    @JvmStatic
    var crashCPStartTime: Long = 0

    @JvmStatic
    var crashCPEndTime: Long = 0

    @JvmStatic
    var apmCPStartTime: Long = 0

    @JvmStatic
    var apmCPEndTime: Long = 0

    @JvmStatic
    var coreCPStartTime: Long = 0

    @JvmStatic
    var coreCPEndTime: Long = 0

    @VisibleForTesting
    var isTracesFlushed = false


    fun flushTraces() {
        if (Instabug.isEnabled() && !isTracesFlushed) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Instabug enabled, flushing launch traces")
            PoolProvider.postIOTask {
                synchronized(this) {
                    coreCPStartTime *= 1000
                    coreCPEndTime *= 1000
                    if (CustomTracesValidator.shouldSaveTrace(DiagnosticsConstants.CORE_PROVIDER)) {
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.CORE_PROVIDER, coreCPStartTime, coreCPEndTime
                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.CORE_PROVIDER} trace executed in ${coreCPEndTime - coreCPStartTime} microseconds"
                        )
                    }

                    crashCPStartTime *= 1000
                    crashCPEndTime *= 1000
                    if (crashCPStartTime != 0L && CustomTracesValidator.shouldSaveTrace(
                            DiagnosticsConstants.CRASH_PROVIDER
                        )
                    ) {
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.CRASH_PROVIDER,
                            crashCPStartTime,
                            crashCPEndTime
                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.CRASH_PROVIDER} trace executed in ${crashCPEndTime - crashCPStartTime} microseconds"
                        )
                    }

                    apmCPStartTime *= 1000
                    apmCPEndTime *= 1000
                    if (apmCPStartTime != 0L && CustomTracesValidator.shouldSaveTrace(
                            DiagnosticsConstants.APM_PROVIDER
                        )
                    ) {
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.APM_PROVIDER, apmCPStartTime, apmCPEndTime
                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.APM_PROVIDER} trace executed in ${apmCPEndTime - apmCPStartTime} microseconds"
                        )
                    }

                    if (CustomTracesValidator.shouldSaveTrace(DiagnosticsConstants.BUILDER_MAIN)) {
                        builderFGStartTime *= 1000
                        builderFGEndTime *= 1000
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.BUILDER_MAIN, getFGStartTime(), getFGEndTime()
                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.BUILDER_MAIN} trace executed in ${getFGEndTime() - getFGStartTime()} microseconds"
                        )
                    }

                    if (builderBGEndTime != 0L && CustomTracesValidator.shouldSaveTrace(
                            DiagnosticsConstants.BUILDER_BG
                        )
                    ) {
                        builderBGStartTime *= 1000
                        IBGDiagnostics.logTrace(
                            DiagnosticsConstants.BUILDER_BG,
                            builderBGStartTime,
                            builderBGEndTime * 1000

                        )
                        InstabugSDKLogger.privateVerbose(
                            Constants.LOG_TAG,
                            "${DiagnosticsConstants.BUILDER_BG} trace executed in ${(builderBGEndTime * 1000) - builderBGStartTime} microseconds"
                        )
                    }
                    isTracesFlushed = true
                }
            }
        }
    }


    @VisibleForTesting
    fun getFGStartTime(): Long = when {
        crashCPStartTime == 0L && apmCPStartTime == 0L -> coreCPStartTime
        crashCPStartTime == 0L -> min(coreCPStartTime, apmCPStartTime)
        apmCPStartTime == 0L -> min(coreCPStartTime, crashCPStartTime)
        else -> min(min(coreCPStartTime, crashCPStartTime), apmCPStartTime)
    }

    @VisibleForTesting
    fun getFGEndTime(): Long {
        var crashCpDuration = 0L
        var apmCpDuration = 0L
        val coreCpDuration = coreCPEndTime - coreCPStartTime
        val builderDuration = builderFGEndTime - builderFGStartTime
        val builderConstructorDuration = builderConstructorEndTime - builderConstructorStartTime

        if (crashCPStartTime != 0L)
            crashCpDuration = crashCPEndTime - crashCPStartTime
        if (apmCPStartTime != 0L)
            apmCpDuration = apmCPEndTime - apmCPStartTime

        return (getFGStartTime() + crashCpDuration + apmCpDuration + coreCpDuration + builderDuration + builderConstructorDuration)
    }

}