package com.instabug.library

import androidx.annotation.WorkerThread
import java.io.File

interface WatchableCacheDirectory {
    /**
     * Adds a watcher by ID to block cleansing if not consented.
     * @param watcherId the watcher id to be added.
     */
    fun addWatcher(watcherId: Int)

    /**
     * Consent on cleansing by a given watcher ID.
     * @param watcherId the watcher id to consider the consent from.
     */
    fun consentOnCleansing(watcherId: Int)

    /**
     * Removes a watcher from the watchers list, hence, stop blocking for its consent.
     * @param watcherId the watcher ID to be removed.
     */
    fun removeWatcher(watcherId: Int)

    /**
     * Query the state of a given watcher by its id.
     * @return a nullable [Boolean] value. If null, means the watcher is not added
     */
    fun queryWatcherConsent(watcherId: Int): Boolean?
}

/**
 * A generic contract for providing caching directories in an organized manner
 */
interface FilesCacheDirectory {
    /**
     * The current saving directory as [File].
     * Could be null if there's no current caching directory.
     */
    val filesDirectory: File?

    @WorkerThread
    fun deleteFilesDirectory() {
        filesDirectory?.takeIf(File::exists)?.deleteRecursively()
    }
}

interface GenericSpansCacheDirectory<T> : FilesCacheDirectory {
    /**
     * The current span directory as [File].
     * Could be null if there's no running span.
     */
    val currentSpanDirectory: T?

    /**
     * The old spans directories saved in the parent directory excluding the currently running
     * span directory.
     */
    val oldSpansDirectories: List<T>

    /**
     * Sets current span id.
     */
    fun setCurrentSpanId(spanId: String?)
}

/**
 * An extension to {FileCacheDirectory} contract for span-based directories.
 */
interface SpansCacheDirectory : GenericSpansCacheDirectory<File>

interface WatchableSpansCacheDirectory : SpansCacheDirectory, WatchableCacheDirectory