package com.instabug.library.util.overairversion

import androidx.annotation.VisibleForTesting
import com.instabug.library.Constants
import com.instabug.library.internal.sharedpreferences.corePref
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.overairversion.OverAirSharedConstants.EXPO_LENGTH_LIMIT
import com.instabug.library.util.overairversion.OverAirSharedConstants.EXPO_LENGTH_LIMIT_EXCEEDED_ERROR
import com.instabug.library.util.overairversion.OverAirSharedConstants.EXPO_NOT_PROVIDED_ERROR
import com.instabug.library.util.overairversion.OverAirSharedConstants.PREF_KEY_IB_OVER_AIR_VERSION
import com.instabug.library.util.overairversion.OverAirSharedConstants.PREF_VALUE_NOT_SET
import com.instabug.library.util.overairversion.OverAirSharedConstants.WITH_EXPO_FORMAT

object ExpoVersionHandler : OnAirVersionTypeHandler {

    @VisibleForTesting
    var storedExpoVersion by corePref(PREF_KEY_IB_OVER_AIR_VERSION, PREF_VALUE_NOT_SET)

    /**
     * Stores the provided Expo version or clear the stored one if the new one is not valid.
     * validation criteria includes not being null or blank (consists only of whitespaces).
     * @param version the newly provided Expo version.
     * @param type which type to detect the corresponding handler if ExpoVersionHandler or CodePushHandler
     * @param isSetByUser always true in case expo version.
     */
    override fun handleVersion(version: String?, type: Int, isSetByUser: Boolean?) {
        version.takeIf { OverAirSharedConstants.isValid(version) }
            ?.let(this::trimAndLogIfNeeded)
            ?.also(this::storeValue)
            ?.also { OverAirSharedConstants.storedType = type }
            ?: clearExpoDataWithLog()
    }

    /**
     * sanitize expo version format to be linked with device provider
     * @param version the provided Expo version.
     */
    override fun sanitize(version: String): String =
        storedExpoVersion.takeUnless { it == PREF_VALUE_NOT_SET }
            ?.let { storedVersion -> WITH_EXPO_FORMAT.format(version, storedVersion) } ?: version

    /**
     * Stores the provided Expo version to be delegate to sharedpref.
     * @param version the provided Expo version.
     */
    private fun storeValue(version: String) {
        storedExpoVersion = version
    }

    /**
     * Trims the provided Expo version if its length exceeds the specified limit (currently 36 chars).
     * In addition, logs an error notifying the user that the provided version exceeds the specified
     * limit and a trimmed version shall be used instead.
     * @param version the provided Expo version
     * @return the [String] provided in the input if its length didn't exceed the specified limit or
     * a [String] represents the first n chars of the input (trimmed version).
     */
    private fun trimAndLogIfNeeded(version: String): String =
        version.trim().let { trimmedVersion ->
            if (trimmedVersion.length <= EXPO_LENGTH_LIMIT) return trimmedVersion
            InstabugSDKLogger.w(Constants.LOG_TAG, EXPO_LENGTH_LIMIT_EXCEEDED_ERROR)
            trimmedVersion.substring(0, EXPO_LENGTH_LIMIT)
        }

    /**
     * Clears the stored Expo version and logs an error notifying the user that there's no
     * Expo version provided and we'll proceed with the default app version if the user has initially
     * used the API to set a Expo version.
     */
    private fun clearExpoDataWithLog() {
        InstabugSDKLogger.w(Constants.LOG_TAG, EXPO_NOT_PROVIDED_ERROR)
        storedExpoVersion = PREF_VALUE_NOT_SET
    }
}
