package com.instabug.library.util;

import androidx.annotation.Keep;

import java.util.concurrent.TimeUnit;

/**
 * A utility class containing time-related helper methods (i.e. unit conversion)
 */
@Keep
public final class TimeUtils {

    public static final long DAY_MILLIS = TimeUnit.DAYS.toMillis(1);
    public static final long MINUTE = 60 * 1000;

    private TimeUtils() {
    }

    /**
     * @param month the number of month to be converted
     * @return time in millis representing that period
     */
    public static long monthToMillis(int month) {
        return TimeUnit.DAYS.toMillis(30L * month);
    }

    /**
     * @param month the number of month to be converted
     * @return time in hours representing that period
     */
    public static long monthToHours(int month) {
        return TimeUnit.DAYS.toHours(30L * month);
    }

    /**
     * @param month the number of month to be converted
     * @return time in days representing that period
     */
    public static long monthToDays(int month) {
        return TimeUnit.DAYS.toDays(30L * month);
    }

    /**
     * @param millis the number of milliseconds to be converted
     * @return time in seconds representing that period
     */
    public static int millisToSeconds(long millis) {
        return (int) TimeUnit.MILLISECONDS.toSeconds(millis);
    }

    /**
     * @param since    the older point of time
     * @param until    the more recent point of time
     * @param timeUnit the unit of {@code since} & {@code until} (i.e. {@link TimeUnit#DAYS}
     * @return the time diff in milliseconds
     */
    public static long elapsedTimeMillis(long since, long until, TimeUnit timeUnit) {
        long sinceMillis = timeUnit.toMillis(since);
        long untilMillis = timeUnit.toMillis(until);
        return untilMillis - sinceMillis;
    }

    /**
     * Helper method for easier mocking
     *
     * @return the current time in milliseconds.
     */
    public static long currentTimeMillis() {
        return System.currentTimeMillis();
    }

    /**
     * Helper method for easier mocking
     *
     * @return the current time in seconds.
     */
    public static long currentTimeSeconds() {
        return System.currentTimeMillis() / 1000;
    }

    public static boolean hasXHoursPassed(long time, long hoursInMillis) {
        return System.currentTimeMillis() - time > hoursInMillis;
    }

    public static long currentTimeStampMicroSeconds() {
        return TimeUnit.MILLISECONDS.toMicros(currentTimeMillis());
    }

    public static long nanoTime() {
        return System.nanoTime();
    }

    public static long microTime() {
        return TimeUnit.NANOSECONDS.toMicros(nanoTime());
    }
}
