package com.instabug.library.util;

import android.content.Context;

import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

final class SdkLocaleRegistry {

    /**
     * This map holds all the currently supported languages by the SDK. For more context refer to IBGSRV-5481
     * <p>
     * The map key is the language code and the value is all the SPECIFICALLY supported variants (if any).
     * If the language is supported by all of its variants the value will be an empty collection
     */
    @VisibleForTesting
    static final Map<String, List<String>> supportedLocales;
    /**
     * We send either the language code (i.e. "en") if we support all variants of this language or
     * we concatenate the country code to it if we support a specific variant of this language (i.e. "pt-BR").
     * Special locales are the ones that do not follow this rule.
     * <p>
     * For example:
     * In Android we support the "no" and on iOS they support "nb". They are ultimately the exact same thing
     * However, this may lead to some inconsistency when we communicate with the BE.
     * So we agreed to map these to some sort of a unified code (in that case "nb-NO").
     * {@code specialLocales} should hold all these languages that have special mapping rule to it.
     * For more context refer to IBGSRV-5481
     */
    private static final Set<String> specialLocales;
    private static final String DEFAULT = "default";

    static {
        // supported
        supportedLocales = new HashMap<>();
        supportedLocales.put("en", getSupportedVariants("en"));
        supportedLocales.put("ar", getSupportedVariants("ar"));
        supportedLocales.put("cs", getSupportedVariants("cs"));
        supportedLocales.put("da", getSupportedVariants("da"));
        supportedLocales.put("de", getSupportedVariants("de"));
        supportedLocales.put("es", getSupportedVariants("es"));
        supportedLocales.put("fr", getSupportedVariants("fr"));
        supportedLocales.put("it", getSupportedVariants("it"));
        supportedLocales.put("ja", getSupportedVariants("ja"));
        supportedLocales.put("ko", getSupportedVariants("ko"));
        supportedLocales.put("nl", getSupportedVariants("nl"));
        supportedLocales.put("no", getSupportedVariants("no"));
        supportedLocales.put("pl", getSupportedVariants("pl"));
        supportedLocales.put("pt", getSupportedVariants("pt"));
        supportedLocales.put("ru", getSupportedVariants("ru"));
        supportedLocales.put("sk", getSupportedVariants("sk"));
        supportedLocales.put("sv", getSupportedVariants("sv"));
        supportedLocales.put("tr", getSupportedVariants("tr"));
        supportedLocales.put("zh", getSupportedVariants("zh"));
        supportedLocales.put("fi", getSupportedVariants("fi"));
        supportedLocales.put("az", getSupportedVariants("az"));
        supportedLocales.put("hu", getSupportedVariants("hu"));
        supportedLocales.put("ca", getSupportedVariants("ca"));
        //special
        specialLocales = new HashSet<>();
        specialLocales.add("no"); // Android
        specialLocales.add("nb"); // iOS
    }

    private final Context context;

    SdkLocaleRegistry(Context context) {
        this.context = context;
    }

    /**
     * This method takes a language code (i.e. "en") and returns all the SPECIFICALLY supported variants
     * by the SDK or an empty collection if all variants are supported
     *
     * @return a set of the supported variants. Empty set means all variants are supported
     */
    @NonNull
    @VisibleForTesting
    static List<String> getSupportedVariants(String languageCode) {
        List<String> languageVariants = new ArrayList<>();
        // DISCLAIMER: the lists order is very critical
        // For more context refer to IBGSRV-5481
        switch (languageCode) {
            case "pt":
                languageVariants.add("PT");
                languageVariants.add("BR");
                return languageVariants;
            case "zh":
                languageVariants.add("CN");
                languageVariants.add("TW");
                return languageVariants;
            case "ca":
                languageVariants.add("ES");
                return languageVariants;
            default:
                return Collections.emptyList();
        }
    }

    /**
     * Returns the current locale identifier as defined in IBGSRV-5481 if the current locale
     * is one of the support languages by the SDK or falls back to the default
     *
     * @return the current locale identifier or "default" as a fallback
     */
    String getCurrentLocaleResolved() {
        Locale locale = InstabugCore.getLocale(context);
        return resolveLocale(locale);
    }

    /**
     * Returns the passed locale identifier as defined in IBGSRV-5481 if it is one of the support
     * languages by the SDK or falls back to the default
     *
     * @param locale the locale do resolved and extract the identifier
     * @return the identifier if the locale is supported by the SDK or "default" as a fallback
     */
    String resolveLocale(Locale locale) {
        String language = locale.getLanguage();
        String country = locale.getCountry();
        return isSupported(language) ? getUnifiedIdentifier(language, country) : DEFAULT;
    }

    /**
     * We send either the language code (i.e. "en") if we support all variants of this language
     * or we concatenate the country code to it if we support a specific variant of this language (i.e. "pt-BR").
     * If this locale variant is not supported return one of the supported variants
     *
     * @param language the lang code (i.e. "en")
     * @param country  the language variant (i.e. "UK")
     * @return the unified locale code as per agreement in IBGSRV-5481
     */
    private String getUnifiedIdentifier(String language, String country) {
        if (isSupported(language) && isSpecialLocale(language)) {
            return getSpecialLocaleUnifiedIdentifier(language);
        }
        if (isSupported(language) && isAllVariantsSupported(language)) {
            return language;
        }
        if (isSupported(language) && isVariantSupported(language, country)) {
            return language + "-" + country;
        }
        if (isSupported(language) && !isVariantSupported(language, country)) {
            return language + "-" + getDefaultSupportedVariant(language);
        }
        return DEFAULT;
    }

    private String getSpecialLocaleUnifiedIdentifier(String language) {
        if (language.equals("no") || language.equals("nb")) {
            return "nb-NO";
        }
        return DEFAULT;
    }

    /**
     * @param language the language code (i.e. "en")
     * @return true if the passed language is within the supported languages by the SDK, false otherwise.
     */
    @VisibleForTesting
    boolean isSupported(String language) {
        return supportedLocales.containsKey(language) || specialLocales.contains(language);
    }

    /**
     * @param language the language code (i.e. "en")
     * @return true if the passed language is supported by its all variants by the SDK, false otherwise.
     */
    @VisibleForTesting
    boolean isAllVariantsSupported(String language) {
        if (!isSupported(language)) {
            return false;
        }
        List<String> supportedVariants = supportedLocales.get(language);
        return supportedVariants == null || supportedVariants.isEmpty();
    }

    /**
     * @param language the language code (i.e. "en")
     * @return true if the passed language does support the passed variant, false otherwise
     */
    @VisibleForTesting
    boolean isVariantSupported(String language, String variant) {
        if (!isSupported(language)) {
            return false;
        }
        List<String> supportedVariants = supportedLocales.get(language);
        return supportedVariants != null && supportedVariants.contains(variant);
    }

    /**
     * @param language the language code (i.e. "en")
     * @return true if the passed language falls within the language that has a special case (i.e. no)
     */
    @VisibleForTesting
    boolean isSpecialLocale(String language) {
        return specialLocales.contains(language);
    }

    private String getDefaultSupportedVariant(String language) {
        List<String> supportedVariants = getSupportedVariants(language);
        if (supportedVariants.size() > 0) {
            return supportedVariants.get(0);
        }
        return "";
    }
}
