package com.instabug.library.sessioncontroller

import com.instabug.library.Constants
import com.instabug.library.SessionManager
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.model.v3Session.SessionEvent
import com.instabug.library.sessionV3.manager.IBGSessionManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.threading.PoolProvider
import java.util.concurrent.locks.ReentrantLock
import kotlin.concurrent.withLock

internal object SessionManualController {

    private val sessionManualControllerConfigs
        get() = SessionManualControllerServiceLocator.sessionManualControllerConfigs
    private val v2SessionManager: SessionManager
        get() = SessionManager.getInstance()
    private val v3SessionManager: IBGSessionManager
        get() = IBGSessionManager

    @Volatile
    private var _sessionLifecycleCallback: IBGSessionLifecycleCallback? = null

    private val isSessionV3Enabled: Boolean
        get() = InstabugCore.isV3SessionEnabled()

    private val reentrantLock =
        ReentrantLock(true)

    @Volatile
    private var v2StartDone = false

    @Volatile
    private var v3StartDone = false

    @Volatile
    private var v2StopDone = false

    @Volatile
    private var v3StopDone = false

    @JvmStatic
    fun startSession() {
        if (!isEnabled()) {
            logErrorMessage("Manual session control feature is disabled, start a new session call is ignored!")
            return
        } else if (InstabugCore.getRunningSession() != null) {
            logErrorMessage("Current session must end before starting a new one, start a new session call is ignored!")
            return
        }
        v2SessionManager.handleSessionStartEvent(true)
        v3SessionManager emit SessionEvent.Start(startedManually = true)
    }

    @JvmStatic
    fun stopSession() {
        if (!isEnabled()) {
            logErrorMessage("Manual session control feature is disabled, end a session call is ignored!")
            return
        } else if (InstabugCore.getRunningSession() == null) {
            logErrorMessage("There is no running session to end, end a session call is ignored!")
            return
        }
        v2SessionManager.stopCurrentSession()
        v3SessionManager emit SessionEvent.Stop()
    }

    @JvmStatic
    fun setSessionLifecycleCallback(sessionLifecycleCallback: IBGSessionLifecycleCallback?) {
        _sessionLifecycleCallback = sessionLifecycleCallback
    }

    @JvmStatic
    fun isEnabled(): Boolean {
        return sessionManualControllerConfigs.controlSessionManuallyEnabled
    }

    private fun logErrorMessage(message: String) {
        InstabugSDKLogger.e(Constants.LOG_TAG, message)
    }

    @JvmStatic
    fun handleCoreEvents(coreEvent: IBGSdkCoreEvent) {
        if (isEnabled()) {
            reentrantLock.withLock {
                handleCoreEventsInternally(coreEvent)
            }
        }
    }

    private fun handleCoreEventsInternally(coreEvent: IBGSdkCoreEvent) {
        when (coreEvent) {
            IBGSdkCoreEvent.Session.SessionStarted -> handleV2SessionStarted()
            IBGSdkCoreEvent.V3Session.V3SessionStarted -> handleV3SessionStarted()
            IBGSdkCoreEvent.Session.SessionFinished -> handleV2SessionFinished()
            IBGSdkCoreEvent.V3Session.V3SessionFinished -> handleV3SessionFinished()
            else -> Unit
        }
    }

    private fun handleV2SessionStarted() {
        if (!isSessionV3Enabled || v3StartDone) {
            fireStart()
            return
        }
        v2StartDone = true
    }

    private fun handleV3SessionStarted() {
        if (v2StartDone) {
            fireStart()
            return
        }
        v3StartDone = true
    }

    private fun handleV2SessionFinished() {
        if (!isSessionV3Enabled || v3StopDone) {
            fireStop()
            return
        }
        v2StopDone = true
    }

    private fun handleV3SessionFinished() {
        if (v2StopDone) {
            fireStop()
            return
        }
        v3StopDone = true
    }

    private fun fireStart() {
        v2StartDone = false
        v3StartDone = false
        runStartedCallBack()
    }

    private fun fireStop() {
        v2StopDone = false
        v3StopDone = false
        runStoppedCallBack()
    }

    private fun runStoppedCallBack() {
        _sessionLifecycleCallback?.let {
            PoolProvider.postMainThreadTask {
                try {
                    it.onSessionStop()
                } catch (t: Throwable) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG,
                        "Exception occurred while running IBGSessionLifecycleCallback#onSessionStop().",
                        t
                    )
                }
            }
        }
    }

    private fun runStartedCallBack() {
        _sessionLifecycleCallback?.let {
            PoolProvider.postMainThreadTask {
                try {
                    it.onSessionStart()
                } catch (t: Throwable) {
                    InstabugSDKLogger.e(
                        Constants.LOG_TAG,
                        "Exception occurred while running IBGSessionLifecycleCallback#onSessionStart().",
                        t
                    )
                }
            }
        }
    }
}
