package com.instabug.library.sessionV3.sync

import com.instabug.library.Constants.LOG_TAG
import com.instabug.library.IBGNetworkWorker
import com.instabug.library.InstabugNetworkJob
import com.instabug.library.model.v3Session.IBGSessionDTO
import com.instabug.library.model.v3Session.IBGSessionMapper.constructRequest
import com.instabug.library.networkv2.RequestResponse
import com.instabug.library.networkv2.request.Request
import com.instabug.library.networkv2.request.RequestType
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator
import com.instabug.library.sessionV3.model.V3SessionSyncResult
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.TimeUtils
import com.instabug.library.util.extenstions.logDebug
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONObject


object IBGSessionsSyncJob : InstabugNetworkJob() {
    private val sessionDataController get() = IBGSessionServiceLocator.sessionDataController
    private val networkManager get() = IBGSessionServiceLocator.networkManager
    private val configurations get() = IBGSessionServiceLocator.sessionConfigurations
    private val v3SessionSyncResultEventBus get() = IBGSessionServiceLocator.v3SessionSyncResultEventBus
    private val rateLimiter
        get() = IBGSessionServiceLocator.rateLimiter { ids ->
            sessionDataController.deleteSessions(ids)
            v3SessionSyncResultEventBus.post(V3SessionSyncResult.RateLimited(ids))
        }
    private const val SR_RATE_LIMITED_KEY = "session_replay_limited"

    private fun getCallbacks(sessionsIds: List<String>) =
        object : Request.Callbacks<RequestResponse, Throwable> {
            override fun onSucceeded(response: RequestResponse?) {
                InstabugSDKLogger.d(
                    LOG_TAG, "$sessionsIds sent successfully ${response?.responseCode}"
                )
                takeIf { isSessionReplayRateLimited(response?.responseBody) }?.let {
                    "Session Replay is rate limited for sessions with ids $sessionsIds"
                        .logDebug(LOG_TAG)

                    v3SessionSyncResultEventBus.post(V3SessionSyncResult.RateLimited(sessionsIds))
                } ?: v3SessionSyncResultEventBus.post(V3SessionSyncResult.Success(sessionsIds))
                rateLimiter.reset()
                sessionDataController.deleteSessions(sessionsIds)
                if (configurations.droppedSessionCount != 0) configurations
                    .droppedSessionCount = 0

                configurations.lastSyncTime = TimeUtils.currentTimeMillis()
                syncBatch()
            }

            override fun onFailed(error: Throwable?) {
                error ?: return
                if (!rateLimiter.inspect(error, sessionsIds))
                    InstabugSDKLogger.e(
                        LOG_TAG,
                        "something went wrong while syncing sessions",
                        error
                    )
            }

        }

    private fun isSessionReplayRateLimited(responseBody: Any?): Boolean {
        return responseBody?.takeIf { it is String }
            ?.let {
                val responseJson = JSONObject(it as String)
                val rateLimited = responseJson.optBoolean(SR_RATE_LIMITED_KEY, false)
                rateLimited
            } ?: false
    }

    private fun syncBatch() {
        sessionDataController
            .collectSessions()
            ?.let(::sync)
    }

    private fun sync(sessions: IBGSessionDTO) = sessions
        .takeUnless { rateLimiter.applyIfPossible(it.sessionsIds) }
        ?.also { "Sessions with sre enabled count ${it.sessionReplayCount}".logDebug(LOG_TAG) }
        ?.constructRequest()
        ?.let { request -> doRequest(request, sessions.sessionsIds) }


    private fun doRequest(
        request: Request, sessionsIds: List<String>
    ) = networkManager.doRequestOnSameThread(
        RequestType.NORMAL, request, getCallbacks(sessionsIds)
    )


    override fun start() {
        enqueueJob(IBGNetworkWorker.CORE) {
            runOrLogError(errorMessage = "Something Went Wrong while syncing Sessions") {
                sessionDataController.deleteSyncedSessions()
                syncBatch()
            }
        }
    }
}
