package com.instabug.library.sessionV3.sync

import androidx.annotation.WorkerThread
import com.instabug.library.model.v3Session.SyncStatus
import com.instabug.library.sessionV3.cache.SessionCacheManager
import com.instabug.library.sessionV3.cache.id
import com.instabug.library.sessionV3.cache.syncStatus
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator

/**
 * evaluate sessions to be synced
 * */
interface IBGSessionBatchEvaluator {
    /**
     * Filters the sessions that have the potential to be synced a marks them as [SyncStatus.READY_FOR_SYNC]
     * if the specified [SessionBatchingFilter] behavior excludes them from filtering.
     * @param batchingFilter Specifies the batching filter behavior.
     * */
    @WorkerThread
    fun evaluate(batchingFilter: SessionBatchingFilter)
}

object IBGSessionBatchEvaluatorImpl : IBGSessionBatchEvaluator {
    private val cacheManager by lazy { IBGSessionServiceLocator.sessionCacheManger }

    override fun evaluate(batchingFilter: SessionBatchingFilter) = run {
        batchingFilter(cacheManager.offlineAndReadySessions)
            .takeIf { toBeSyncedSessions -> toBeSyncedSessions.isNotEmpty() }
            ?.filter { session -> session.syncStatus == SyncStatus.OFFLINE }
            ?.map { session -> session.id }
            ?.also(this::markOfflineAsReadyForSync); Unit
    }

    private val SessionCacheManager.offlineAndReadySessions
        get() = querySessionsIdsBySyncStatus(SyncStatus.OFFLINE, SyncStatus.READY_FOR_SYNC)

    private fun markOfflineAsReadyForSync(offlineSessionsIds: List<String>) =
        offlineSessionsIds.takeIf { it.isNotEmpty() }?.let {
            cacheManager.changeSyncStatus(
                from = SyncStatus.OFFLINE,
                to = SyncStatus.READY_FOR_SYNC,
                sessionsIds = it
            )
        }
}
