package com.instabug.library.sessionV3.cache

import com.instabug.library.featuresflags.managers.FeatureFlagsManager
import com.instabug.library.featuresflags.mappers.asJsonObject
import com.instabug.library.featuresflags.mappers.toSessionsFeaturesFlags
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionFeaturesFlagsEntry.COLUMN_FEATURES_FLAGS_ARRAY
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionFeaturesFlagsEntry.COLUMN_SESSION_SERIAL
import com.instabug.library.internal.storage.cache.dbv2.IBGDbContract.SessionFeaturesFlagsEntry.TABLE_NAME
import com.instabug.library.internal.storage.cache.dbv2.IBGDbManager
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg
import com.instabug.library.internal.storage.cache.dbv2.asArgs
import com.instabug.library.internal.storage.cache.dbv2.joinToArgs
import com.instabug.library.internal.storage.cache.dbv2.kQuery
import com.instabug.library.util.extenstions.runOrLogError
import org.json.JSONArray

/**
 * Manage the added features flags for sessions.
 */
interface SessionFeaturesFlagsCacheManager {

    /**
     * Fetch the features flags for the supplied sessions serials.
     * @param sessionsSerials to get the relevant features flags for each one of them.
     */
    fun getFeaturesFlags(sessionsSerials: List<Long>): Map<Long, JSONArray>

    /**
     * Save currently existing features flags with the current session.
     * @param sessionSerial for the currently ended session.
     */
    fun saveFeaturesFlags(sessionSerial: Long)
}

internal class SessionFeaturesFlagsCacheManagerImpl(
    private val featureFlagsManager: FeatureFlagsManager,
    private val database: IBGDbManager
) : SessionFeaturesFlagsCacheManager {
    override fun getFeaturesFlags(sessionsSerials: List<Long>): Map<Long, JSONArray> {
        return database.runOrLogError(errorMessage = "Something went wrong while querying features flags") {
            kQuery(
                table = TABLE_NAME,
                whereClause = whereSessionSerialIn(sessionsSerials)
            )?.toSessionsFeaturesFlags()
        }.getOrNull() ?: emptyMap()
    }

    override fun saveFeaturesFlags(sessionSerial: Long) {
        featureFlagsManager.getAllFeaturesFlags(1.0f)?.takeIf { it.isNotEmpty() }?.filterNotNull()
            ?.let { featuresFlags ->
                val featuresFlagsAsJson = featuresFlags.map { it.asJsonObject() }
                val featuresFlagsContentValues = IBGContentValues().apply {
                    put(COLUMN_SESSION_SERIAL, sessionSerial, true)
                    put(COLUMN_FEATURES_FLAGS_ARRAY, featuresFlagsAsJson.toString(), false)
                }
                database.runOrLogError(errorMessage = "Something went wrong while inserting session features flags") {
                    insertWithOnConflictReplace(TABLE_NAME, null, featuresFlagsContentValues)
                }
            } ?: database.runOrLogError(errorMessage = "Something went wrong while clearing session features flags") {
            val whereClause = "$COLUMN_SESSION_SERIAL=? "
            val whereArgs: MutableList<IBGWhereArg> = ArrayList()
            whereArgs.add(IBGWhereArg(sessionSerial.toString(), true))
            this.delete(TABLE_NAME, whereClause, whereArgs)
        }
    }
}

private fun whereSessionSerialIn(sessionsSerials: List<Long>) =
    ("$COLUMN_SESSION_SERIAL IN ${sessionsSerials.joinToArgs()}"
            to sessionsSerials.map { it.toString() }.asArgs())